<?php
declare(strict_types=1);

/**
 * فانکشن‌های کمکی پروژه.
 */

/** فایل لاگ را اگر وجود ندارد می‌سازد تا خطاها بدون 403 ذخیره شوند. */
function ensure_log_file(string $path): void
{
    if (!is_file($path)) {
        @touch($path);
    }
}

/**
 * ثبت رویداد در فایل لاگ با تاریخ شمسی و میلادی برای بررسی سریع.
 */
function bot_log(string $filePath, string $message, array $context = []): void
{
    ensure_log_file($filePath);
    $time = new DateTimeImmutable('now', new DateTimeZone('Asia/Tehran'));
    $jsonContext = $context === [] ? '' : ' ' . json_encode($context, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    file_put_contents($filePath, sprintf("[%s] %s%s%s", $time->format('Y-m-d H:i:s'), $message, $jsonContext, PHP_EOL), FILE_APPEND);
}

/**
 * تمیز کردن داده‌های ورودی برای جلوگیری از اسکریپت‌ها.
 */
function clean_string(?string $value): string
{
    return trim(strip_tags((string) ($value ?? '')));
}

/**
 * بررسی می‌کند آیا ورودی فقط از اعداد تشکیل شده است یا نه.
 */
function is_valid_code(string $value): bool
{
    return preg_match('/^\d{1,10}$/', $value) === 1;
}

/**
 * ساخت کیبورد اینلاین برای پیام‌های ربات.
 */
function make_inline_keyboard(array $buttons): string
{
    return json_encode(['inline_keyboard' => $buttons], JSON_UNESCAPED_UNICODE);
}

function format_duration_label(int $seconds): string
{
    if ($seconds <= 0) {
        return 'همیشه بماند';
    }

    if ($seconds < 60) {
        return sprintf('%d ثانیه', $seconds);
    }

    if ($seconds % 3600 === 0) {
        $hours = (int) ($seconds / 3600);
        return sprintf('%d ساعت', $hours);
    }

    if ($seconds % 60 === 0) {
        $minutes = (int) ($seconds / 60);
        return sprintf('%d دقیقه', $minutes);
    }

    $minutes = intdiv($seconds, 60);
    $remaining = $seconds % 60;
    $parts = [];
    if ($minutes > 0) {
        $parts[] = sprintf('%d دقیقه', $minutes);
    }
    if ($remaining > 0) {
        $parts[] = sprintf('%d ثانیه', $remaining);
    }

    return implode(' و ', $parts);
}

/**
 * ساخت متن وضعیت برای پیام‌های ارسال همگانی.
 */
function build_broadcast_progress_text(array $job): string
{
    $status = $job['status'] ?? 'pending';
    $sent = (int) ($job['sent_count'] ?? 0);
    $failed = (int) ($job['failed_count'] ?? 0);
    $total = (int) ($job['total_count'] ?? 0);
    $pending = max(0, $total - $sent - $failed);

    $statusLine = 'ارسال همگانی در صف انتظار ⏳';
    $liveLine = '';

    if ($status === 'completed') {
        $statusLine = 'ارسال همگانی با موفقیت تمام شد 🎉';
    } elseif ($status === 'cancelled') {
        $statusLine = 'ارسال همگانی متوقف شد ❌';
    } elseif ($status === 'running') {
        $statusLine = 'ارسال همگانی در حال انجام … ⏱️';
        $liveLine = '🟢 وضعیت: آنلاین';
    } elseif ($status === 'pending') {
        $statusLine = 'ارسال همگانی در صف انتظار ⏳';
        $liveLine = '🟡 وضعیت: آماده‌سازی';
    }

    $ttl = (int) ($job['auto_delete_seconds'] ?? 0);
    $pin = (int) ($job['pin_message'] ?? 0) === 1;

    $parts = [
        $statusLine,
    ];

    if ($liveLine !== '') {
        $parts[] = $liveLine;
    }

    $parts[] = sprintf("✅ ارسال موفق: %s کاربر", number_format($sent));
    $parts[] = sprintf("❌ ارسال ناموفق: %s کاربر", number_format($failed));
    $parts[] = sprintf("⏳ در انتظار: %s کاربر", number_format($pending));
    $parts[] = sprintf("⏱ حذف خودکار: %s", format_duration_label($ttl));
    $parts[] = sprintf("📌 سنجاق پیام: %s", $pin ? 'بله' : 'خیر');

    return implode("\n", $parts);
}

/**
 * ساخت کیبورد وضعیت ارسال همگانی بر اساس اتمام یا ادامه کار.
 */
function build_broadcast_progress_keyboard(int $jobId, bool $finished): string
{
    if ($finished) {
        return make_inline_keyboard([
            [[
                'text' => 'بازگشت ↩️',
                'callback_data' => 'admin_menu',
            ]],
        ]);
    }

    return make_inline_keyboard([
        [[
            'text' => 'لغو ارسال همگانی ❌',
            'callback_data' => sprintf('broadcast_cancel:%d', $jobId),
        ]],
        [[
            'text' => 'بازگشت ↩️',
            'callback_data' => 'admin_menu',
        ]],
    ]);
}

/**
 * ارسال پاسخ JSON استاندارد به کلاینت‌های وب.
 */
function json_response(array $data, int $status = 200): void
{
    http_response_code($status);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
}

/** تشخیص اینکه کاربر فعلی از مدیران اصلی است یا خیر. */
function is_admin_user(array $telegramConfig, array $chat): bool
{
    $chatId = (int) ($chat['id'] ?? 0);
    return $chatId !== 0 && in_array($chatId, $telegramConfig['sudo_ids'] ?? [], true);
}



