<?php
declare(strict_types=1);

// -----------------------------------------------------------------------------
// وبهوک اصلی ربات آپلودر. این فایل را در هاستینگر به عنوان index.php قرار بده.
// -----------------------------------------------------------------------------

require __DIR__ . '/helpers.php';
require __DIR__ . '/database.php';
require __DIR__ . '/telegram_api.php';

$config = require __DIR__ . '/config.php';

date_default_timezone_set($config['app']['timezone']);

ensure_log_file(__DIR__ . '/php-error.log');
ini_set('log_errors', '1');
ini_set('error_log', __DIR__ . '/php-error.log');

$logFile = $config['app']['log_file'];
ensure_log_file($logFile);

// دریافت بدنه خام درخواست.
$rawInput = file_get_contents('php://input');
if ($rawInput === false || $rawInput === '') {
    bot_log($logFile, 'empty webhook');
    json_response(['ok' => false, 'message' => 'empty payload'], 400);
    return;
}

try {
    $update = json_decode($rawInput, true, 512, JSON_THROW_ON_ERROR);
} catch (Throwable $exception) {
    bot_log($logFile, 'invalid json', ['error' => $exception->getMessage()]);
    json_response(['ok' => false, 'message' => 'bad json'], 400);
    return;
}

try {
    $pdo = Database::pdo($config['database']);
} catch (Throwable $exception) {
    bot_log($logFile, 'database connection failed', ['error' => $exception->getMessage()]);
    json_response(['ok' => false, 'message' => 'db error'], 500);
    return;
}

if (isset($update['message'])) {
    handle_message_update($pdo, $config, $update['message']);
} elseif (isset($update['callback_query'])) {
    handle_callback_query($pdo, $config, $update['callback_query']);
} elseif (isset($update['chat_member'])) {
    handle_chat_member_update($pdo, $config, $update['chat_member']);
} elseif (isset($update['my_chat_member'])) {
    handle_my_chat_member_update($pdo, $config, $update['my_chat_member']);
}

json_response(['ok' => true]);

// -----------------------------------------------------------------------------
// توابع مربوط به پردازش آپدیت‌ها
// -----------------------------------------------------------------------------

/**
 * مدیریت پیام‌های متنی کاربران.
 */
function handle_message_update(PDO $pdo, array $config, array $message): void
{
    $token = $config['telegram']['bot_token'];
    $chat = $message['chat'] ?? [];
    $chatId = (int) ($chat['id'] ?? 0);

    if ($chatId === 0) {
        return;
    }

    // ثبت کاربر در دیتابیس برای آمار دقیق.
    $botUser = upsert_bot_user($pdo, $chat);
    $botUserId = (int) ($botUser['id'] ?? 0);
    $isBlocked = (bool) ($botUser['is_blocked'] ?? false);

    $isAdmin = is_admin_user($config['telegram'], $chat);

    if ($isAdmin && process_admin_update($pdo, $config, $message)) {
        return;
    }

    if ($isBlocked) {
        $notice = "دسترسی شما به ربات محدود شده است ❌";
        telegram_send_message($token, $chatId, $notice);
        return;
    }

    if (!$isAdmin && !enforce_force_channels($pdo, $config, $chat, $botUser)) {
        return;
    }

    if (!empty($message['text'])) {
        $text = trim($message['text']);

        if (strpos($text, '/start') === 0) {
            $code = trim(str_replace('/start', '', $text));
            if ($code !== '') {
                $code = ltrim($code, '=');
                serve_media_code($pdo, $config, $chatId, $code, $botUserId);
                return;
            }

            send_welcome($token, $chat, $config);
            return;
        }

        if (is_valid_code($text)) {
            serve_media_code($pdo, $config, $chatId, $text, $botUserId);
            return;
        }

        telegram_send_message($token, $chatId, "کد درست نیست عزیزم 😅\nیه عدد درست مثل 114613 بفرست تا آهنگ‌هاتو برات بفرستم 💽");
    }
}

/**
 * پاسخ‌گویی به دکمه‌های اینلاین.
 */
function handle_callback_query(PDO $pdo, array $config, array $callback): void
{
    $token = $config['telegram']['bot_token'];
    $queryId = $callback['id'];
    $chatId = (int) ($callback['message']['chat']['id'] ?? 0);
    $isAdmin = is_admin_user($config['telegram'], $callback['from'] ?? []);

    bot_log($config['app']['log_file'], 'callback_received', [
        'data' => $callback['data'] ?? '',
        'chat_id' => $chatId,
        'is_admin' => $isAdmin,
    ]);

    try {
        telegram_request($token, 'answerCallbackQuery', [
            'callback_query_id' => $queryId,
        ]);
    } catch (RuntimeException $exception) {
        $message = $exception->getMessage();
        if (
            str_contains($message, 'query is too old') ||
            str_contains($message, 'query ID is invalid')
        ) {
            bot_log($config['app']['log_file'], 'callback_expired', [
                'chat_id' => $chatId,
                'query_id' => $queryId,
                'error' => $message,
            ]);
        } else {
            throw $exception;
        }
    }

    if ($isAdmin && process_admin_callback($pdo, $config, $callback)) {
        return;
    }

    if (!$isAdmin && process_user_callback($pdo, $config, $callback)) {
        return;
    }
}

function handle_chat_member_update(PDO $pdo, array $config, array $update): void
{
    $chat = $update['chat'] ?? [];
    $chatType = $chat['type'] ?? '';
    if (!in_array($chatType, ['channel', 'supergroup'], true)) {
        return;
    }


    $chatId = (int) ($chat['id'] ?? 0);
    if ($chatId === 0) {
        return;
    }

    $userDebug = $update['new_chat_member']['user'] ?? [];
    bot_log($config['app']['log_file'], 'chat_member_update', [
        'chat_id' => $chatId,
        'new_status' => $update['new_chat_member']['status'] ?? null,
        'old_status' => $update['old_chat_member']['status'] ?? null,
        'invite_link' => $update['invite_link']['invite_link'] ?? null,
        'user_id' => $userDebug['id'] ?? null,
        'username' => $userDebug['username'] ?? null,
    ]);

    $newMember = $update['new_chat_member'] ?? [];
    $oldMember = $update['old_chat_member'] ?? [];
    $newStatus = $newMember['status'] ?? '';
    $oldStatus = $oldMember['status'] ?? '';

    $channel = fetch_force_channel_by_chat($pdo, $chatId);
    if (!$channel || (int) ($channel['is_active'] ?? 0) !== 1) {
        return;
    }

    $joinedStatuses = ['member', 'administrator', 'creator'];
    $leftStatuses = ['left', 'kicked'];

    if (in_array($newStatus, $joinedStatuses, true) && !in_array($oldStatus, $joinedStatuses, true)) {
        $invite = $update['invite_link']['invite_link'] ?? '';
        $storedInvite = $channel['invite_link'] ?? '';
        if (($channel['chat_type'] ?? '') !== 'bot') {
            if ($storedInvite !== '') {
                if ($invite === '') {
                    bot_log($config['app']['log_file'], 'force_invite_missing', [
                        'channel_id' => $channel['id'] ?? null,
                        'user_id' => $newMember['user']['id'] ?? null,
                        'update' => $update,
                    ]);
                    return;
                }
                if (!hash_equals($storedInvite, $invite)) {
                    bot_log($config['app']['log_file'], 'force_invite_mismatch', [
                        'channel_id' => $channel['id'] ?? null,
                        'expected' => $storedInvite,
                        'actual' => $invite,
                        'user_id' => $newMember['user']['id'] ?? null,
                        'update' => $update,
                    ]);
                    return;
                }
            } else {
                ensure_force_channel_invite_link($pdo, $config, $channel);
                $channel = fetch_force_channel($pdo, (int) $channel['id']);
                if ($channel && !empty($channel['invite_link'])) {
                    $storedInvite = $channel['invite_link'];
                    if ($invite === '') {
                        bot_log($config['app']['log_file'], 'force_invite_missing_after_generate', [
                            'channel_id' => $channel['id'] ?? null,
                            'user_id' => $newMember['user']['id'] ?? null,
                            'update' => $update,
                        ]);
                        return;
                    }
                    if (!hash_equals($storedInvite, $invite)) {
                        bot_log($config['app']['log_file'], 'force_invite_mismatch_after_generate', [
                            'channel_id' => $channel['id'] ?? null,
                            'expected' => $storedInvite,
                            'actual' => $invite,
                            'user_id' => $newMember['user']['id'] ?? null,
                            'update' => $update,
                        ]);
                        return;
                    }
                }
            }
        }

        $userInfo = $newMember['user'] ?? null;
        $telegramId = (int) ($userInfo['id'] ?? 0);
        $botUser = $telegramId > 0 ? find_bot_user_by_telegram_id($pdo, $telegramId) : null;
        if (!$botUser) {
            bot_log($config['app']['log_file'], 'force_join_user_unknown', [
                'channel_id' => $channel['id'] ?? null,
                'user_id' => $telegramId,
            ]);
            return;
        }

        register_force_channel_join($pdo, $config, $channel, $botUser);

        try {
            $updatedChannel = fetch_force_channel($pdo, (int) $channel['id']);
            if ($updatedChannel) {
                finalize_force_channel_if_completed($pdo, $config, $updatedChannel);
            }
        } catch (Throwable $exception) {
            bot_log($config['app']['log_file'], 'chat_member_finalize_failed', [
                'channel_id' => $channel['id'] ?? null,
                'error' => $exception->getMessage(),
            ]);
        }
        return;
    }

    if (in_array($newStatus, $leftStatuses, true) && in_array($oldStatus, $joinedStatuses, true)) {
        $userInfo = $oldMember['user'] ?? $newMember['user'] ?? [];
        register_force_channel_leave($pdo, $config, $channel, $userInfo);
        return;
    }
}

function handle_my_chat_member_update(PDO $pdo, array $config, array $update): void
{
    $chat = $update['chat'] ?? [];
    if (($chat['type'] ?? '') !== 'private') {
        return;
    }

    $chatId = (int) ($chat['id'] ?? 0);
    if ($chatId === 0) {
        return;
    }

    $from = $update['from'] ?? [];
    $profile = [
        'id' => $chatId,
        'username' => $chat['username'] ?? ($from['username'] ?? ''),
        'first_name' => $chat['first_name'] ?? ($from['first_name'] ?? ''),
        'last_name' => $chat['last_name'] ?? ($from['last_name'] ?? ''),
    ];

    $botUser = upsert_bot_user($pdo, $profile);
    $userId = (int) ($botUser['id'] ?? 0);
    if ($userId === 0) {
        return;
    }

    $newStatus = $update['new_chat_member']['status'] ?? '';

    if (in_array($newStatus, ['kicked', 'left'], true)) {
        set_user_bot_block_status($pdo, $userId, true);
        record_activity($pdo, (string) $chatId, 'user_blocked_bot', [
            'status' => $newStatus,
        ]);
    } elseif ($newStatus === 'member') {
        set_user_bot_block_status($pdo, $userId, false);
        record_activity($pdo, (string) $chatId, 'user_unblocked_bot', [
            'status' => $newStatus,
        ]);
    }
}

/**
 * پیام خوشامد با لحن دوستانه.
 */
function send_welcome(string $token, array $chat, array $config): void
{
    $chatId = (int) $chat['id'];
    $replyMarkup = null;

    if (is_admin_user($config['telegram'], $chat)) {
        $replyMarkup = make_inline_keyboard([
            [[
                'text' => 'پنل مدیریت 🎛️',
                'callback_data' => 'admin_menu',
            ]],
        ]);
    }

    telegram_send_message(
        $token,
        $chatId,
        "سلام رفیق 💚\nبرای دانلود آهنگ فقط شماره کد رو بفرست؛ مثلا 114613\nمنم سریع برات همون پلی‌لیست یا ترک رو می‌فرستم ✨",
        $replyMarkup ? ['reply_markup' => $replyMarkup] : []
    );
}

/**
 * سرویس‌دهی کدهای رسانه‌ای.
 */
function serve_media_code(PDO $pdo, array $config, int $chatId, string $code, int $botUserId = 0): void
{
    if (!is_valid_code($code)) {
        telegram_send_message($config['telegram']['bot_token'], $chatId, "این کد معتبر نیست 😕 دوباره چک کن.");
        return;
    }

    $token = $config['telegram']['bot_token'];

    $stmt = $pdo->prepare(
        'SELECT code, media_type, file_id, caption, performer, title, position, download_count
         FROM media_library
         WHERE code = :code
         ORDER BY position ASC'
    );
    $stmt->execute([':code' => $code]);
    $files = $stmt->fetchAll();

    if ($files === []) {
        telegram_send_message($token, $chatId, "کدی که فرستادی دستم نرسید 😔\nاگر مطمئنی درسته، یه بار دیگه امتحان کن یا از پشتیبانی کمک بگیر.");
        return;
    }

    telegram_send_action($token, $chatId, 'upload_document');

    foreach ($files as $item) {
        try {
            send_media_item($token, $chatId, $item);
        } catch (Throwable $exception) {
            bot_log($config['app']['log_file'], 'send media failed', [
                'error' => $exception->getMessage(),
                'code' => $code,
            ]);
        }
    }

    $pdo->prepare('UPDATE media_library SET download_count = download_count + 1 WHERE code = :code AND position = 1')
        ->execute([':code' => $code]);

    $pdo->prepare('INSERT INTO activity_logs (actor, action, details) VALUES (:actor, :action, :details)')->execute([
        ':actor' => (string) $chatId,
        ':action' => 'user_download',
        ':details' => json_encode(['code' => $code], JSON_UNESCAPED_UNICODE),
    ]);

    telegram_send_message($token, $chatId, "تمام شد! امیدوارم خوشت بیاد 🌈\nاگر دوست داشتی برای رفیقات هم بفرست 💌");
}

/**
 * ارسال فایل بر اساس نوع ذخیره شده.
 */
function send_media_item(string $token, int $chatId, array $item): void
{
    $caption = $item['caption'] ?? '';

    switch ($item['media_type']) {
        case 'audio':
            telegram_request($token, 'sendAudio', [
                'chat_id' => $chatId,
                'audio' => $item['file_id'],
                'caption' => $caption,
                'title' => $item['title'] ?? null,
                'performer' => $item['performer'] ?? null,
            ]);
            break;
        case 'video':
            telegram_request($token, 'sendVideo', [
                'chat_id' => $chatId,
                'video' => $item['file_id'],
                'caption' => $caption,
            ]);
            break;
        case 'photo':
            telegram_request($token, 'sendPhoto', [
                'chat_id' => $chatId,
                'photo' => $item['file_id'],
                'caption' => $caption,
            ]);
            break;
        default:
            telegram_request($token, 'sendDocument', [
                'chat_id' => $chatId,
                'document' => $item['file_id'],
                'caption' => $caption,
            ]);
    }
}

/**
 * ثبت یا به‌روزرسانی کاربر در جدول bot_users.
 */
function upsert_bot_user(PDO $pdo, array $chat): array
{
    $telegramId = (int) ($chat['id'] ?? 0);
    if ($telegramId === 0) {
        return [
            'id' => 0,
            'telegram_id' => 0,
            'is_blocked' => false,
        ];
    }

    $stmt = $pdo->prepare('SELECT id FROM bot_users WHERE telegram_id = :telegram_id LIMIT 1');
    $stmt->execute([':telegram_id' => $telegramId]);
    $userId = $stmt->fetchColumn();

    $username = clean_string($chat['username'] ?? '');
    $firstName = clean_string($chat['first_name'] ?? '');
    $lastName = clean_string($chat['last_name'] ?? '');
    $combinedName = trim(($firstName . ' ' . $lastName));
    if ($combinedName === '') {
        $combinedName = $firstName !== '' ? $firstName : $lastName;
    }
    if ($combinedName === '' && $username !== '') {
        $combinedName = '@' . ltrim($username, '@');
    }
    if ($combinedName === '') {
        $combinedName = (string) $telegramId;
    }

    $data = [
        ':telegram_id' => $telegramId,
        ':username' => $username !== '' ? ltrim($username, '@') : null,
        ':name' => $combinedName,
    ];

    if ($userId) {
        $pdo->prepare('UPDATE bot_users SET username = :username, name = :name, last_seen_at = NOW() WHERE telegram_id = :telegram_id')->execute($data);
    } else {
        $pdo->prepare('INSERT INTO bot_users (telegram_id, username, name) VALUES (:telegram_id, :username, :name)')->execute($data);
        $userId = $pdo->lastInsertId();
    }

    $select = $pdo->prepare('SELECT id, telegram_id, username, name, joined_at, last_seen_at, is_blocked, has_blocked_bot FROM bot_users WHERE telegram_id = :telegram_id LIMIT 1');
    $select->execute([':telegram_id' => $telegramId]);
    $row = $select->fetch();

    if (!$row) {
        return [
            'id' => 0,
            'telegram_id' => $telegramId,
            'is_blocked' => false,
        ];
    }

    $row['is_blocked'] = (bool) ($row['is_blocked'] ?? false);
    $row['has_blocked_bot'] = (bool) ($row['has_blocked_bot'] ?? false);
    return $row;
}

// -----------------------------------------------------------------------------
// مدیریت ادمین‌ها داخل خود ربات
// -----------------------------------------------------------------------------

function process_admin_update(PDO $pdo, array $config, array $message): bool
{
    $token = $config['telegram']['bot_token'];
    $chat = $message['chat'] ?? [];
    $chatId = (int) ($chat['id'] ?? 0);
    if ($chatId === 0) {
        return false;
    }

    ensure_admin_record($pdo, $chat);
    $state = get_admin_state($pdo, $chatId);

    $text = isset($message['text']) ? trim($message['text']) : '';
    $mediaInfo = extract_media_info($message);

    if ($state['pending_mode'] === 'broadcast_message') {
        return handle_admin_broadcast_message($pdo, $config, $message, $state);
    }

    if ($state['pending_mode'] === 'broadcast_filter') {
        telegram_send_message($token, $chatId, 'اول یکی از فیلترها رو از دکمه‌های پایین انتخاب کن 🔽');
        return true;
    }

    if ($state['pending_mode'] === 'broadcast_options') {
        telegram_send_message($token, $chatId, 'مدت نگهداری پیام و سنجاق شدن را با دکمه‌ها مشخص کن ⏱');
        return true;
    }

    if ($text !== '') {
        $command = strtolower(strtok($text, " \t\n"));
        if ($command !== false && isset($command[0]) && $command[0] === '/' && !in_array($command, ['/panel', '/admin'], true)) {
            return false;
        }
    }

    if ($state['pending_mode'] === 'user_message') {
        if ($text === '') {
            telegram_send_message($token, $chatId, "پیامت رو بنویس تا همون رو برای کاربر بفرستم ✉️");
            return true;
        }

        $userId = (int) ($state['pending_code'] ?? 0);
        $target = fetch_user_by_id($pdo, $userId);
        if (!$target) {
            telegram_send_message($token, $chatId, "کاربر پیدا نشد. دوباره از منوی مدیریت کاربران امتحان کن.");
            reset_admin_state($pdo, $chatId);
            send_admin_user_intro($config, $chatId);
            return true;
        }

        $targetChatId = (int) ($target['telegram_id'] ?? 0);
        try {
            telegram_send_message($token, $targetChatId, "پیامی از ادمین:\n\n{$text}");
            telegram_send_message($token, $chatId, "پیام ارسال شد ✅", [
                'reply_markup' => make_inline_keyboard([
                    [[
                        'text' => 'بازگشت ↩️',
                        'callback_data' => 'admin_user_menu',
                    ]],
                ]),
            ]);
        } catch (Throwable $exception) {
            telegram_send_message($token, $chatId, "ارسال پیام ناموفق بود 😔\nخطا: {$exception->getMessage()}");
        }

        record_activity($pdo, (string) $chatId, 'admin_message_user', [
            'user_id' => $targetChatId,
        ]);

        set_admin_state($pdo, $chatId, [
            'pending_mode' => 'user_manage',
            'pending_code' => $userId,
        ]);
        send_user_management_menu($config, $chatId, $target);
        return true;
    }

    if ($state['pending_mode'] === 'user_lookup') {
        if ($text !== '' && $text[0] === '/') {
            return false;
        }
        $resolved = resolve_admin_user_target($pdo, $message, $text);
        if ($resolved === null) {
            telegram_send_message($token, $chatId, "کاربر پیدا نشد 😕\nیوزرنیم (با @) یا آی‌دی عددی رو درست بفرست یا پیامش رو فوروارد کن.");
            return true;
        }

        set_admin_state($pdo, $chatId, [
            'pending_mode' => 'user_manage',
            'pending_code' => $resolved['id'],
            'pending_group_id' => (string) $resolved['telegram_id'],
        ]);
        send_user_management_menu($config, $chatId, $resolved);
        return true;
    }

    if ($state['pending_mode'] === 'force_add') {
        $channel = resolve_force_channel_reference($config, $message, $text);
        if ($channel === null) {
            telegram_send_message($token, $chatId, "کانال پیدا نشد 😕\nلطفاً یک پست از کانال یا گروه رو فوروارد کن یا آیدی دقیقش رو بفرست.");
            return true;
        }

        if (isset($channel['error'])) {
            if ($channel['error'] === 'invite_only') {
                telegram_send_message($token, $chatId, "برای گروه یا کانال خصوصی، حتماً یک پست را فوروارد کن تا بتوانم شناسه را پیدا کنم.");
            } else {
                telegram_send_message($token, $chatId, "این نوع چت پشتیبانی نمی‌شود. لطفاً کانال یا سوپرگروه دیگری ارسال کن.");
            }
            return true;
        }

        if (empty($channel['chat_id'])) {
            telegram_send_message($token, $chatId, "کانال پیدا نشد 😕\nلطفاً یک پست از کانال یا گروه رو فوروارد کن یا آیدی دقیقش رو بفرست.");
            return true;
        }

        try {
            $record = save_force_channel($pdo, $channel, $chatId);
            if (is_force_channel_bot($record)) {
                set_admin_state($pdo, $chatId, [
                    'pending_mode' => 'force_intro',
                    'pending_code' => null,
                    'pending_group_id' => null,
                ]);
                $usernameDisplay = $record['username'] ?? '';
                if ($usernameDisplay !== '') {
                    $usernameDisplay = '@' . ltrim($usernameDisplay, '@');
                } else {
                    $usernameDisplay = clean_string($record['title'] ?? 'ربات');
                }
                telegram_send_message($token, $chatId, sprintf("%s به لیست تبلیغات اضافه شد ✅", $usernameDisplay));
                send_force_channels_menu($pdo, $config, $chatId);
            } else {
                set_admin_state($pdo, $chatId, [
                    'pending_mode' => 'force_verify',
                    'pending_code' => $record['id'],
                    'pending_group_id' => (string) $record['chat_id'],
                ]);
                send_force_channel_card($config, $chatId, $record, 'کانال ذخیره شد ✅');
            }
        } catch (Throwable $exception) {
            telegram_send_message($token, $chatId, "ذخیره کانال با خطا روبه‌رو شد ❗️\n" . $exception->getMessage());
        }
        return true;
    }

    if ($state['pending_mode'] === 'force_verify') {
        $channelId = (int) ($state['pending_code'] ?? 0);
        if ($channelId === 0) {
            send_force_channels_menu($pdo, $config, $chatId);
            return true;
        }

        $link = extract_force_channel_link($text);
        if ($link) {
            update_force_channel_invite_link($pdo, $channelId, $link);
            $record = fetch_force_channel($pdo, $channelId);
            if ($record) {
                send_force_channel_card($config, $chatId, $record, 'لینک جدید ذخیره شد ✅');
            }
            return true;
        }

        if ($text !== '') {
            telegram_send_message($token, $chatId, "اگر لینک خصوصی کانال رو داری همینجا بفرست یا از دکمه‌ها استفاده کن.");
            return true;
        }
    }

    if ($state['pending_mode'] === 'force_target') {
        $channelId = (int) ($state['pending_code'] ?? 0);
        if ($channelId === 0) {
            send_force_channels_menu($pdo, $config, $chatId);
            return true;
        }

        $channel = fetch_force_channel($pdo, $channelId);
        if (!$channel) {
            send_force_channels_menu($pdo, $config, $chatId);
            return true;
        }

        if (!preg_match('/^\d+$/', $text)) {
            telegram_send_message($token, $chatId, "یک عدد مثبت برای تعداد اعضای مورد نیاز بفرست.");
            return true;
        }

        $target = (int) $text;
        if ($target <= 0) {
            telegram_send_message($token, $chatId, "تعداد باید بیشتر از صفر باشد.");
            return true;
        }

        $inviteLink = create_force_channel_invite_link($pdo, $config, $channel, $target);
        if (!$inviteLink) {
            telegram_send_message($token, $chatId, "ساخت لینک دعوت با خطا روبه‌رو شد. مطمئن شو ربات اجازه ساخت لینک دارد و دوباره امتحان کن.");
            return true;
        }

        $updated = fetch_force_channel($pdo, $channelId);
        if ($updated) {
            send_force_channel_card($config, $chatId, $updated, 'لینک اختصاصی ساخته شد ✅');
            send_force_channels_menu($pdo, $config, $chatId);
        }

        set_admin_state($pdo, $chatId, [
            'pending_mode' => 'force_intro',
            'pending_code' => null,
            'pending_group_id' => null,
        ]);
        return true;
    }

    if ($mediaInfo !== null) {
        if ($state['pending_mode'] === 'single') {
            return handle_admin_single_upload($pdo, $config, $chatId, $mediaInfo);
        }

        if ($state['pending_mode'] === 'batch') {
            return handle_admin_batch_media($pdo, $config, $chatId, $mediaInfo, $state);
        }

        telegram_send_message($token, $chatId, "اول از منوی مدیریت گزینه آپلود رو انتخاب کن تا بفهمم باید با رسانه‌هات چکار کنم ✨");
        return true;
    }

    if ($text === '') {
        return false;
    }

    if ($text === '/panel' || $text === '/admin') {
        set_admin_state($pdo, $chatId, [
            'pending_mode' => 'admin_home',
            'pending_code' => null,
            'pending_position' => 1,
            'pending_group_id' => null,
        ]);
        send_admin_menu($config, $chatId);
        return true;
    }

    if (strpos($text, '/done') === 0) {
        finalize_batch_upload($pdo, $config, $chatId, $state);
        return true;
    }

    if ($state['pending_mode'] === 'batch') {
        telegram_send_message($token, $chatId, "دارم آهنگ‌های این سری رو ذخیره می‌کنم 🎶 وقتی تموم شد دکمه «پایان آپلود ✅» رو بزن.");
        return true;
    }

    if ($state['pending_mode'] === 'delete') {
        handle_admin_delete_code($pdo, $config, $chatId, $text);
        return true;
    }

    if ($state['pending_mode'] === 'info') {
        handle_admin_info_code($pdo, $config, $chatId, $text);
        return true;
    }

    if ($state['pending_mode'] === 'user_manage') {
        telegram_send_message($token, $chatId, "برای مدیریت کاربر از دکمه‌های منو استفاده کن یا «بازگشت» رو بزن.");
        return true;
    }

    return false;
}

function process_admin_callback(PDO $pdo, array $config, array $callback): bool
{
    $token = $config['telegram']['bot_token'];
    $chat = $callback['message']['chat'] ?? [];
    $chatId = (int) ($chat['id'] ?? 0);
    $from = $callback['from'] ?? $chat;
    ensure_admin_record($pdo, $from);
    $data = $callback['data'] ?? '';

    switch ($data) {
        case 'admin_menu':
            set_admin_state($pdo, $chatId, [
                'pending_mode' => 'admin_home',
                'pending_code' => null,
                'pending_position' => 1,
                'pending_group_id' => null,
            ]);
            send_admin_menu($config, $chatId);
            return true;

        case 'admin_manage':
            reset_admin_state($pdo, $chatId);
            send_admin_management_menu($config, $chatId);
            return true;

        case 'admin_user_menu':
            $state = get_admin_state($pdo, $chatId);
            $userId = (int) ($state['pending_code'] ?? 0);
            if ($state['pending_mode'] === 'user_manage' && $userId > 0) {
                $user = fetch_user_by_id($pdo, $userId);
                if ($user) {
                    set_admin_state($pdo, $chatId, [
                        'pending_mode' => 'user_manage',
                        'pending_code' => $userId,
                        'pending_group_id' => $state['pending_group_id'] ?? null,
                    ]);
                    send_user_management_menu($config, $chatId, $user);
                    return true;
                }
            }

            set_admin_state($pdo, $chatId, [
                'pending_mode' => 'user_lookup',
                'pending_code' => null,
                'pending_position' => 1,
                'pending_group_id' => null,
            ]);
            send_admin_user_intro($config, $chatId);
            return true;

        case 'admin_back':
            $state = get_admin_state($pdo, $chatId);
            if (($state['pending_mode'] ?? '') === 'admin_home') {
                reset_admin_state($pdo, $chatId);
                $chatInfo = $callback['message']['chat'] ?? ['id' => $chatId];
                send_welcome($token, $chatInfo, $config);
                return true;
            }

            set_admin_state($pdo, $chatId, [
                'pending_mode' => 'admin_home',
                'pending_code' => null,
                'pending_position' => 1,
                'pending_group_id' => null,
            ]);
            send_admin_menu($config, $chatId);
            return true;

        case 'admin_upload_menu':
            reset_admin_state($pdo, $chatId);
            send_admin_upload_menu($config, $chatId);
            return true;

        case 'admin_single':
            set_admin_state($pdo, $chatId, [
                'pending_mode' => 'single',
                'pending_code' => null,
                'pending_position' => 1,
                'pending_group_id' => null,
            ]);
            telegram_send_message($token, $chatId, "برای آپلود تکی فقط همینجا فایل رو بفرست 🎯\nهمینکه ارسال شد، برات یه کد جدید می‌سازم و توی گروه مدیریتی ذخیره می‌کنم.");
            return true;

        case 'admin_batch':
            set_admin_state($pdo, $chatId, [
                'pending_mode' => 'batch',
                'pending_code' => null,
                'pending_position' => 1,
                'pending_group_id' => null,
            ]);
            telegram_send_message($token, $chatId, "شروع کن به ارسال ترک‌ها یا ویدیوهای پشت‌سرهم 🎧\nهر موردی که می‌فرستی ذخیره می‌کنم، وقتی آماده بودی دکمه «پایان آپلود ✅» رو بزن.", [
                'reply_markup' => make_inline_keyboard([
                    [[
                        'text' => 'پایان آپلود ✅',
                        'callback_data' => 'admin_finish',
                    ]],
                    [[
                        'text' => 'بازگشت ↩️',
                        'callback_data' => 'admin_upload_menu',
                    ]],
                ]),
            ]);
            return true;

        case 'admin_finish':
            $state = get_admin_state($pdo, $chatId);
            finalize_batch_upload($pdo, $config, $chatId, $state);
            return true;

        case 'admin_stats':
            send_admin_stats($pdo, $config, $chatId);
            return true;

        case 'admin_delete':
            set_admin_state($pdo, $chatId, [
                'pending_mode' => 'delete',
                'pending_code' => null,
                'pending_position' => 1,
                'pending_group_id' => null,
            ]);
            telegram_send_message($token, $chatId, "کد رسانه‌ای که می‌خوای پاک بشه رو برام بفرست 🗑️");
            return true;

        case 'admin_info':
            set_admin_state($pdo, $chatId, [
                'pending_mode' => 'info',
                'pending_code' => null,
                'pending_position' => 1,
                'pending_group_id' => null,
            ]);
            telegram_send_message($token, $chatId, "کد رسانه‌ای که می‌خوای گزارشش رو ببینی برام بفرست 📊");
            return true;

        case 'admin_user_block':
            $state = get_admin_state($pdo, $chatId);
            $userId = (int) ($state['pending_code'] ?? 0);
            if ($userId === 0) {
                send_admin_user_intro($config, $chatId);
                return true;
            }
            set_user_block_status($pdo, $userId, true);
            $user = fetch_user_by_id($pdo, $userId);
            $userDisplay = format_user_display($user ?? ['telegram_id' => $userId]);
            telegram_send_message($token, $chatId, "کاربر {$userDisplay} مسدود شد ⛔️");
            record_activity($pdo, (string) $chatId, 'admin_block_user', [
                'user_id' => $user['telegram_id'] ?? $userId,
            ]);
            set_admin_state($pdo, $chatId, [
                'pending_mode' => 'user_manage',
                'pending_code' => $userId,
            ]);
            if ($user) {
                send_user_management_menu($config, $chatId, $user);
            }
            return true;

        case 'admin_user_unblock':
            $state = get_admin_state($pdo, $chatId);
            $userId = (int) ($state['pending_code'] ?? 0);
            if ($userId === 0) {
                send_admin_user_intro($config, $chatId);
                return true;
            }
            set_user_block_status($pdo, $userId, false);
            $user = fetch_user_by_id($pdo, $userId);
            $userDisplay = format_user_display($user);
            telegram_send_message($token, $chatId, "کاربر {$userDisplay} آزاد شد ✅", [
                'reply_markup' => make_inline_keyboard([
                    [[
                        'text' => 'بازگشت ↩️',
                        'callback_data' => 'admin_user_menu',
                    ]],
                ]),
            ]);
            record_activity($pdo, (string) $chatId, 'admin_unblock_user', [
                'user_id' => $user['telegram_id'] ?? $userId,
            ]);
            set_admin_state($pdo, $chatId, [
                'pending_mode' => 'user_manage',
                'pending_code' => $userId,
            ]);
            if ($user) {
                send_user_management_menu($config, $chatId, $user);
            }
            return true;

        case 'admin_user_message':
            $state = get_admin_state($pdo, $chatId);
            $userId = (int) ($state['pending_code'] ?? 0);
            if ($userId === 0) {
                send_admin_user_intro($config, $chatId);
                return true;
            }
            set_admin_state($pdo, $chatId, [
                'pending_mode' => 'user_message',
                'pending_code' => $userId,
                'pending_group_id' => $state['pending_group_id'] ?? null,
            ]);
            telegram_send_message($token, $chatId, "پیامی که می‌خوای برای کاربر بفرستم رو بنویس ✉️");
            return true;

        case 'admin_user_stats':
            $state = get_admin_state($pdo, $chatId);
            $userId = (int) ($state['pending_code'] ?? 0);
            if ($userId === 0) {
                 send_admin_user_intro($config, $chatId);
                 return true;
             }
             $user = fetch_user_by_id($pdo, $userId);
             if (!$user) {
                 telegram_send_message($token, $chatId, "کاربر پیدا نشد. از ابتدا دوباره انتخاب کن.");
                 send_admin_user_intro($config, $chatId);
                 return true;
             }
             set_admin_state($pdo, $chatId, [
                 'pending_mode' => 'user_manage',
                 'pending_code' => $userId,
                 'pending_group_id' => $state['pending_group_id'] ?? null,
             ]);
             send_user_stats_report($pdo, $config, $chatId, $user);
             return true;

        case 'admin_force_menu':
            set_admin_state($pdo, $chatId, [
                'pending_mode' => 'force_intro',
                'pending_code' => null,
                'pending_position' => 1,
                'pending_group_id' => null,
            ]);
            send_force_channels_menu($pdo, $config, $chatId);
            return true;

        case 'force_add_start':
            set_admin_state($pdo, $chatId, [
                'pending_mode' => 'force_add',
                'pending_code' => null,
                'pending_position' => 1,
                'pending_group_id' => null,
            ]);
            telegram_send_message($token, $chatId, "یک پست از کانال اسپانسر رو فوروارد کن یا آیدی/لینک رو بفرست تا اضافه کنم.");
            return true;

        case 'force_list':
            telegram_request($token, 'answerCallbackQuery', [
                'callback_query_id' => $callback['id'],
                'text' => 'این گزینه دیگر در دسترس نیست.',
                'show_alert' => true,
            ]);
            return true;

        case 'admin_user_menu':
            $state = get_admin_state($pdo, $chatId);
            $userId = (int) ($state['pending_code'] ?? 0);
            if ($state['pending_mode'] === 'user_manage' && $userId > 0) {
                $user = fetch_user_by_id($pdo, $userId);
                if ($user) {
                    set_admin_state($pdo, $chatId, [
                        'pending_mode' => 'user_manage',
                        'pending_code' => $userId,
                        'pending_group_id' => $state['pending_group_id'] ?? null,
                    ]);
                    send_user_management_menu($config, $chatId, $user);
                    return true;
                }
            }

            set_admin_state($pdo, $chatId, [
                'pending_mode' => 'user_lookup',
                'pending_code' => null,
                'pending_position' => 1,
                'pending_group_id' => null,
            ]);
            send_admin_user_intro($config, $chatId);
            return true;

        case 'admin_broadcast':
            set_admin_state($pdo, $chatId, [
                'pending_mode' => 'broadcast_filter',
                'pending_code' => null,
                'pending_position' => 1,
                'pending_group_id' => null,
            ]);
            send_broadcast_filter_menu($config, $chatId);
            return true;
    }

    if (preg_match('/^force_verify:(\d+)$/', $data, $match)) {
        $channelId = (int) $match[1];
        $channel = fetch_force_channel($pdo, $channelId);
        if (!$channel) {
            telegram_request($token, 'answerCallbackQuery', [
                'callback_query_id' => $callback['id'],
                'text' => 'کانال پیدا نشد.',
                'show_alert' => true,
            ]);
            return true;
        }

        if (is_force_channel_bot($channel)) {
            set_admin_state($pdo, $chatId, [
                'pending_mode' => 'force_intro',
                'pending_code' => null,
                'pending_group_id' => null,
            ]);
            $messageId = (int) ($callback['message']['message_id'] ?? 0);
            $statusMessage = 'این مورد مربوط به ربات تبلیغاتی است ✅';
            if ($messageId) {
                edit_force_channel_card($config, $chatId, $messageId, $channel, $statusMessage);
            } else {
                send_force_channel_card($config, $chatId, $channel, $statusMessage);
            }
            telegram_request($token, 'answerCallbackQuery', [
                'callback_query_id' => $callback['id'],
                'text' => 'این مورد یک ربات تبلیغاتی است ✅',
            ]);
            return true;
        }

        $isAdmin = verify_bot_admin_in_channel($config, $channel);
        update_force_channel_admin_status($pdo, $channelId, $isAdmin);
        $updated = fetch_force_channel($pdo, $channelId) ?? $channel;
        $inviteCreated = false;
        if ($isAdmin) {
            $beforeInvite = $updated['invite_link'] ?? '';
            $updated = ensure_force_channel_invite_link($pdo, $config, $updated);
            $inviteCreated = ($beforeInvite === '' && ($updated['invite_link'] ?? '') !== '');
        }
        if (!$isAdmin) {
            $nextMode = 'force_verify';
        } elseif ((int) ($updated['target_joins'] ?? 0) === 0) {
            $nextMode = 'force_target';
        } else {
            $nextMode = 'force_intro';
        }
        set_admin_state($pdo, $chatId, [
            'pending_mode' => $nextMode,
            'pending_code' => $channelId,
            'pending_group_id' => (string) $channel['chat_id'],
        ]);

        $statusMessage = $isAdmin ? 'همه چی درسته! ربات الان ادمین این کاناله ✅' : 'ربات هنوز ادمین نشده ❗️ لطفاً به عنوان ادمین اضافه کن و دوباره امتحان کن.';
        if ($inviteCreated) {
            $statusMessage = 'لینک اختصاصی ساخته شد ✅\n' . $statusMessage;
        }

        $messageId = (int) ($callback['message']['message_id'] ?? 0);
        if ($messageId) {
            edit_force_channel_card($config, $chatId, $messageId, $updated, $statusMessage);
        } else {
            send_force_channel_card($config, $chatId, $updated, $statusMessage);
        }

        telegram_request($token, 'answerCallbackQuery', [
            'callback_query_id' => $callback['id'],
            'text' => $isAdmin ? 'وضعیت کانال به‌روز شد ✅' : 'هنوز ادمین نشده‌ای! لطفاً ربات را ادمین کن.',
            'show_alert' => !$isAdmin,
        ]);

        if ($isAdmin && (int) ($updated['target_joins'] ?? 0) === 0) {
            send_force_goal_prompt($config, $chatId, $channelId);
        }

        return true;
    }

    if (preg_match('/^force_toggle:(\d+):(0|1)$/', $data, $match)) {
        $channelId = (int) $match[1];
        $active = (int) $match[2] === 1;
        $channel = fetch_force_channel($pdo, $channelId);
        if ($channel && is_force_channel_bot($channel)) {
            telegram_request($token, 'answerCallbackQuery', [
                'callback_query_id' => $callback['id'],
                'text' => 'این گزینه فقط برای کانال‌هاست.',
                'show_alert' => true,
            ]);
            return true;
        }

        toggle_force_channel_status($pdo, $channelId, $active);
        if ($active) {
            reset_force_channel_progress($pdo, $channelId);
        }
        $channel = fetch_force_channel($pdo, $channelId);
        if ($channel) {
            $statusMessage = $active ? 'کانال فعال شد ✅' : 'کانال غیرفعال شد ⛔️';
            $messageId = (int) ($callback['message']['message_id'] ?? 0);
            if ($messageId) {
                edit_force_channel_card($config, $chatId, $messageId, $channel, $statusMessage);
            } else {
                send_force_channel_card($config, $chatId, $channel, $statusMessage);
            }
            if ($active) {
                set_admin_state($pdo, $chatId, [
                    'pending_mode' => 'force_target',
                    'pending_code' => $channelId,
                    'pending_group_id' => (string) ($channel['chat_id'] ?? ''),
                ]);
                send_force_goal_prompt($config, $chatId, $channelId);
            }
        }

        telegram_request($token, 'answerCallbackQuery', [
            'callback_query_id' => $callback['id'],
            'text' => $active ? 'کانال فعال شد ✅' : 'کانال غیرفعال شد ⛔️',
        ]);
        return true;
    }

    if (preg_match('/^force_delete:(\d+)$/', $data, $match)) {
        $channelId = (int) $match[1];
        $removed = delete_force_channel($pdo, $config, $channelId);
        if ($removed) {
            $title = clean_string($removed['title'] ?? 'کانال');
            telegram_send_message($token, $chatId, sprintf('کانال %s از لیست اسپانسر حذف شد ❌', $title));
        } else {
            telegram_send_message($token, $chatId, 'کانال پیدا نشد یا قبلاً حذف شده بود.');
        }

        send_force_channels_menu($pdo, $config, $chatId);
        return true;
    }

    if (preg_match('/^force_show:(\d+)$/', $data, $match)) {
        $channelId = (int) $match[1];
        $channel = fetch_force_channel($pdo, $channelId);
        if (!$channel) {
            telegram_request($token, 'answerCallbackQuery', [
                'callback_query_id' => $callback['id'],
                'text' => 'کانال پیدا نشد.',
                'show_alert' => true,
            ]);
            return true;
        }

        if (is_force_channel_bot($channel)) {
            set_admin_state($pdo, $chatId, [
                'pending_mode' => 'force_intro',
                'pending_code' => null,
                'pending_group_id' => null,
            ]);
            send_force_channel_card($config, $chatId, $channel, 'جزئیات ربات باز شد ✅');
            telegram_request($token, 'answerCallbackQuery', [
                'callback_query_id' => $callback['id'],
            ]);
            return true;
        }

        $inviteCreated = false;
        if (!(int) ($channel['bot_is_admin'] ?? 0)) {
            $isAdminNow = verify_bot_admin_in_channel($config, $channel);
            if ($isAdminNow) {
                update_force_channel_admin_status($pdo, $channelId, true);
                $channel = fetch_force_channel($pdo, $channelId) ?? $channel;
            }
        }

        if ((int) ($channel['bot_is_admin'] ?? 0) === 1) {
            try {
                $channel = synchronize_force_channel_counters($pdo, $config, $channel);
                $channel = finalize_force_channel_if_completed($pdo, $config, $channel);
                if ($channel === null) {
                    telegram_send_message($config['telegram']['bot_token'], $chatId, 'کمپین به پایان رسید و کانال از لیست پاک شد ✅');
                    send_force_channels_menu($pdo, $config, $chatId);
                    return true;
                }
            } catch (Throwable $exception) {
                bot_log($config['app']['log_file'], 'force_show_sync_failed', [
                    'channel_id' => $channelId,
                    'error' => $exception->getMessage(),
                ]);
            }
        }

        $targetValue = (int) ($channel['target_joins'] ?? 0);
        $isAdmin = (int) ($channel['bot_is_admin'] ?? 0) === 1;
        $pendingMode = 'force_intro';
        if (!$isAdmin) {
            $pendingMode = 'force_verify';
        } elseif ($targetValue <= 0) {
            $pendingMode = 'force_target';
        }
 
        set_admin_state($pdo, $chatId, [
            'pending_mode' => $pendingMode,
            'pending_code' => $channelId,
            'pending_group_id' => (string) ($channel['chat_id'] ?? ''),
        ]);
 
        $statusMessage = 'جزئیات کانال باز شد ✅';
        if ($inviteCreated) {
            $statusMessage = "لینک اختصاصی ساخته شد ✅\n" . $statusMessage;
        }

        send_force_channel_card($config, $chatId, $channel, $statusMessage);
 
        if ($pendingMode === 'force_target') {
            send_force_goal_prompt($config, $chatId, $channelId, $targetValue > 0);
        }
 
        telegram_request($token, 'answerCallbackQuery', [
            'callback_query_id' => $callback['id'],
        ]);
        return true;
    }

    if (preg_match('/^force_edit_target:(\d+)$/', $data, $match)) {
        $channelId = (int) $match[1];
        $channel = fetch_force_channel($pdo, $channelId);
        if (!$channel) {
            telegram_request($token, 'answerCallbackQuery', [
                'callback_query_id' => $callback['id'],
                'text' => 'کانال پیدا نشد.',
                'show_alert' => true,
            ]);
            return true;
        }

        if (is_force_channel_bot($channel)) {
            telegram_request($token, 'answerCallbackQuery', [
                'callback_query_id' => $callback['id'],
                'text' => 'برای ربات تبلیغاتی نیازی به هدف نیست.',
                'show_alert' => true,
            ]);
            return true;
        }

        set_admin_state($pdo, $chatId, [
            'pending_mode' => 'force_target',
            'pending_code' => $channelId,
            'pending_group_id' => (string) ($channel['chat_id'] ?? ''),
        ]);
        send_force_goal_prompt($config, $chatId, $channelId, true);
        telegram_request($token, 'answerCallbackQuery', [
            'callback_query_id' => $callback['id'],
        ]);
        return true;
    }

    if (preg_match('/^force_stat:(\d+)$/', $data, $match)) {
        $channelId = (int) $match[1];
        $channel = fetch_force_channel($pdo, $channelId);
        if ($channel && is_force_channel_bot($channel)) {
            telegram_request($token, 'answerCallbackQuery', [
                'callback_query_id' => $callback['id'],
                'text' => 'برای ربات تبلیغاتی آماری ثبت نمی‌شود.',
                'show_alert' => true,
            ]);
            return true;
        }
        $message = 'اطلاعاتی ثبت نشده است.';
        if ($channel) {
            $message = sprintf("اعضای فعال: %d\nکل جذب‌شده: %d\nترک کرده‌اند: %d",
                (int) ($channel['current_joins'] ?? 0),
                (int) ($channel['total_joins'] ?? 0),
                (int) ($channel['total_left'] ?? 0)
            );
        }

        telegram_request($token, 'answerCallbackQuery', [
            'callback_query_id' => $callback['id'],
            'text' => $message,
            'show_alert' => true,
        ]);
        return true;
    }

    if (preg_match('/^force_refresh:(\d+)$/', $data, $match)) {
        $channelId = (int) $match[1];
        telegram_request($token, 'answerCallbackQuery', [
            'callback_query_id' => $callback['id'],
        ]);

        $channel = fetch_force_channel($pdo, $channelId);
        if ($channel && is_force_channel_bot($channel)) {
            telegram_send_message($token, $chatId, 'این گزینه فقط برای کانال‌ها فعال است.');
            return true;
        }
        if ($channel) {
            try {
                $channel = synchronize_force_channel_counters($pdo, $config, $channel);
                $channel = finalize_force_channel_if_completed($pdo, $config, $channel);
                if ($channel === null) {
                    telegram_send_message($config['telegram']['bot_token'], $chatId, 'کمپین به پایان رسید و کانال از لیست پاک شد ✅');
                    send_force_channels_menu($pdo, $config, $chatId);
                    return true;
                }
                $messageId = (int) ($callback['message']['message_id'] ?? 0);
                if ($messageId) {
                    edit_force_channel_card($config, $chatId, $messageId, $channel, 'آمار به‌روزرسانی شد ✅');
                } else {
                    send_force_channel_card($config, $chatId, $channel, 'آمار به‌روزرسانی شد ✅');
                }
            } catch (Throwable $exception) {
                bot_log($config['app']['log_file'], 'force_refresh_failed', [
                    'channel_id' => $channelId,
                    'error' => $exception->getMessage(),
                ]);
            }
        }
        return true;
    }

    if ($data === 'broadcast_abort') {
        reset_admin_state($pdo, $chatId);
        telegram_send_message($token, $chatId, 'ارسال همگانی لغو شد ❌');
        send_admin_menu($config, $chatId);
        return true;
    }

    if (preg_match('/^broadcast_filter:(all|day|week|month)$/', $data, $match)) {
        $filter = $match[1];
        $defaultTtl = 30;
        $defaultPin = 0;
        set_admin_state($pdo, $chatId, [
            'pending_mode' => 'broadcast_options',
            'pending_code' => $defaultTtl,
            'pending_position' => $defaultPin,
            'pending_group_id' => $filter,
        ]);

        telegram_request($token, 'answerCallbackQuery', [
            'callback_query_id' => $callback['id'],
        ]);

        send_broadcast_options_menu($config, $chatId, $filter, $defaultTtl, (bool) $defaultPin);
        return true;
    }

    if (preg_match('/^broadcast_ttl:(\d+)$/', $data, $match)) {
        $ttl = max(0, (int) $match[1]);
        $state = get_admin_state($pdo, $chatId);
        $filter = $state['pending_group_id'] ?? 'all';
        $pin = (int) ($state['pending_position'] ?? 0) === 1;
        set_admin_state($pdo, $chatId, [
            'pending_code' => $ttl,
        ]);

        telegram_request($token, 'answerCallbackQuery', [
            'callback_query_id' => $callback['id'],
        ]);

        $messageId = (int) ($callback['message']['message_id'] ?? 0);
        send_broadcast_options_menu($config, $chatId, $filter, $ttl, $pin, $messageId ?: null);
        return true;
    }

    if ($data === 'broadcast_pin:toggle') {
        $state = get_admin_state($pdo, $chatId);
        $filter = $state['pending_group_id'] ?? 'all';
        $ttl = (int) ($state['pending_code'] ?? 0);
        $current = (int) ($state['pending_position'] ?? 0);
        $new = $current === 1 ? 0 : 1;
        set_admin_state($pdo, $chatId, [
            'pending_position' => $new,
        ]);

        telegram_request($token, 'answerCallbackQuery', [
            'callback_query_id' => $callback['id'],
        ]);

        $messageId = (int) ($callback['message']['message_id'] ?? 0);
        send_broadcast_options_menu($config, $chatId, $filter, $ttl, $new === 1, $messageId ?: null);
        return true;
    }

    if ($data === 'broadcast_options_continue') {
        $state = get_admin_state($pdo, $chatId);
        $filter = $state['pending_group_id'] ?? 'all';
        $ttl = (int) ($state['pending_code'] ?? 0);
        $pin = (int) ($state['pending_position'] ?? 0) === 1;
        set_admin_state($pdo, $chatId, [
            'pending_mode' => 'broadcast_message',
        ]);

        telegram_request($token, 'answerCallbackQuery', [
            'callback_query_id' => $callback['id'],
        ]);

        send_broadcast_message_prompt($config, $chatId, $filter, $ttl, $pin);
        return true;
    }

    if ($data === 'broadcast_options_back') {
        $state = get_admin_state($pdo, $chatId);
        $currentMode = $state['pending_mode'] ?? 'broadcast_filter';
        $filter = $state['pending_group_id'] ?? 'all';
        $ttl = (int) ($state['pending_code'] ?? 0);
        $pin = (int) ($state['pending_position'] ?? 0) === 1;

        telegram_request($token, 'answerCallbackQuery', [
            'callback_query_id' => $callback['id'],
        ]);

        if ($currentMode === 'broadcast_message') {
            set_admin_state($pdo, $chatId, [
                'pending_mode' => 'broadcast_options',
            ]);
            $messageId = (int) ($callback['message']['message_id'] ?? 0);
            send_broadcast_options_menu($config, $chatId, $filter, $ttl, $pin, $messageId ?: null);
        } else {
            set_admin_state($pdo, $chatId, [
                'pending_mode' => 'broadcast_filter',
                'pending_code' => null,
                'pending_position' => 1,
                'pending_group_id' => null,
            ]);
            send_broadcast_filter_menu($config, $chatId);
        }
        return true;
    }

    if (preg_match('/^broadcast_cancel:(\d+)$/', $data, $match)) {
        $jobId = (int) $match[1];
        $updated = cancel_broadcast_job($pdo, $jobId);

        telegram_request($token, 'answerCallbackQuery', [
            'callback_query_id' => $callback['id'],
            'text' => $updated ? 'ارسال متوقف شد ❌' : 'این ارسال قبلاً تمام شده است.',
            'show_alert' => !$updated,
        ]);

        if ($updated) {
            update_broadcast_progress_message($pdo, $config, $jobId, true);
        }

        return true;
    }

    return false;
}

function send_admin_menu(array $config, int $chatId): void
{
    $keyboard = make_inline_keyboard([
        [[
            'text' => 'افزودن رسانه 🎶',
            'callback_data' => 'admin_upload_menu',
        ]],
        [[
            'text' => 'مدیریت رسانه 🗂️',
            'callback_data' => 'admin_manage',
        ]],
        [[
            'text' => 'کانال‌های اسپانسر 📣',
            'callback_data' => 'admin_force_menu',
        ]],
        [[
            'text' => 'مدیریت کاربران 👥',
            'callback_data' => 'admin_user_menu',
        ]],
        [[
            'text' => 'ارسال همگانی 📢',
            'callback_data' => 'admin_broadcast',
        ]],
        [[
            'text' => 'آمار 📊',
            'callback_data' => 'admin_stats',
        ], [
            'text' => 'بازگشت ↩️',
            'callback_data' => 'admin_back',
        ]],
    ]);

    try {
        $response = telegram_send_message(
            $config['telegram']['bot_token'],
            $chatId,
            "سلام رئیس 👑 امروز چی کار کنیم؟ گزینه مورد نظر رو انتخاب کن.",
            ['reply_markup' => $keyboard]
        );
        bot_log($config['app']['log_file'], 'admin_menu_sent', [
            'chat_id' => $chatId,
            'message_id' => $response['message_id'] ?? null,
        ]);
    } catch (Throwable $exception) {
        bot_log($config['app']['log_file'], 'admin_menu_failed', [
            'chat_id' => $chatId,
            'error' => $exception->getMessage(),
        ]);
    }
}

function send_admin_upload_menu(array $config, int $chatId): void
{
    $keyboard = make_inline_keyboard([
        [[
            'text' => 'آپلود تکی 🎤',
            'callback_data' => 'admin_single',
        ]],
        [[
            'text' => 'آپلود چندتایی 🎹',
            'callback_data' => 'admin_batch',
        ]],
        [[
            'text' => 'بازگشت ↩️',
            'callback_data' => 'admin_menu',
        ]],
    ]);

    $response = telegram_send_message(
        $config['telegram']['bot_token'],
        $chatId,
        "کدوم روش رو می‌خوای انتخاب کنی؟ تکی یا چندتایی؟",
        ['reply_markup' => $keyboard]
    );

    bot_log($config['app']['log_file'], 'admin_upload_menu_sent', [
        'chat_id' => $chatId,
        'message_id' => $response['message_id'] ?? null,
    ]);
}

function send_admin_management_menu(array $config, int $chatId): void
{
    $keyboard = make_inline_keyboard([
        [[
            'text' => 'حذف رسانه 🗑️',
            'callback_data' => 'admin_delete',
        ]],
        [[
            'text' => 'گزارش رسانه 📈',
            'callback_data' => 'admin_info',
        ]],
        [[
            'text' => 'بازگشت ↩️',
            'callback_data' => 'admin_menu',
        ]],
    ]);

    telegram_send_message(
        $config['telegram']['bot_token'],
        $chatId,
        "اینجا می‌تونی رسانه‌ها رو مدیریت کنی؛ حذف یا گزارش بگیر. هر کدوم رو که می‌خوای انتخاب کن.",
        ['reply_markup' => $keyboard]
    );
}

function send_admin_user_intro(array $config, int $chatId): void
{
    telegram_send_message(
        $config['telegram']['bot_token'],
        $chatId,
        "برای مدیریت کاربر یکی از این کارها رو انجام بده:\n• پیام کاربر رو فوروارد کن\n• آی‌دی عددی کاربر رو بفرست (مثلاً 5529726664)\n• یوزرنیم با یا بدون @ بفرست\nبعدش می‌تونی مسدود کنی، پیام بدی یا گزارش بگیری.",
        [
            'reply_markup' => make_inline_keyboard([
                [[
                    'text' => 'بازگشت ↩️',
                    'callback_data' => 'admin_menu',
                ]],
            ]),
        ]
    );
}

function send_user_management_menu(array $config, int $chatId, array $user): void
{
    $token = $config['telegram']['bot_token'];
    $userDisplay = format_user_display($user);
    $status = $user['is_blocked'] ? "وضعیت: مسدود ❌" : "وضعیت: فعال ✅";
    $botBlockStatus = !empty($user['has_blocked_bot']) ? "ربات را بلاک کرده: بله ❗️" : "ربات را بلاک کرده: خیر ✅";
    $joinedAt = format_datetime_for_display($user['joined_at'] ?? null, $config);
    $lastSeen = format_datetime_for_display($user['last_seen_at'] ?? null, $config);

    $buttons = [
        [[
            'text' => $user['is_blocked'] ? 'رفع مسدودیت ✅' : 'مسدود کردن ⛔️',
            'callback_data' => $user['is_blocked'] ? 'admin_user_unblock' : 'admin_user_block',
        ]],
        [[
            'text' => 'ارسال پیام ✉️',
            'callback_data' => 'admin_user_message',
        ]],
        [[
            'text' => 'گزارش کاربر 📊',
            'callback_data' => 'admin_user_stats',
        ]],
        [[
            'text' => 'بازگشت ↩️',
            'callback_data' => 'admin_user_menu',
        ]],
    ];

    telegram_send_message(
        $token,
        $chatId,
        "{$userDisplay}\n{$status}\n{$botBlockStatus}\nتاریخ عضویت: {$joinedAt}\nآخرین فعالیت: {$lastSeen}",
        ['reply_markup' => make_inline_keyboard($buttons)]
    );
}

function resolve_admin_user_target(PDO $pdo, array $message, string $text): ?array
{
    if (isset($message['forward_from']['id'])) {
        $forward = $message['forward_from'];
        $chatMock = [
            'id' => $forward['id'],
            'username' => $forward['username'] ?? '',
            'first_name' => $forward['first_name'] ?? '',
            'last_name' => $forward['last_name'] ?? '',
        ];
        return upsert_bot_user($pdo, $chatMock);
    }

    if ($text === '') {
        return null;
    }

    $parts = preg_split('/\s+/', trim($text));
    $telegramId = null;
    $username = null;

    foreach ($parts as $part) {
        if ($part === '') {
            continue;
        }

        if (preg_match('/^-?\d+$/', $part)) {
            $telegramId = (int) $part;
        } elseif ($part[0] === '@') {
            $username = substr($part, 1);
        } elseif (preg_match('/^[A-Za-z0-9_]{3,}$/', $part)) {
            $username = $part;
        }
    }

    if ($telegramId !== null) {
        $user = fetch_user_by_telegram($pdo, $telegramId);
        if ($user !== null) {
            return $user;
        }
    }

    if ($username !== null) {
        $user = fetch_user_by_username($pdo, $username);
        if ($user !== null) {
            return $user;
        }
    }

    return null;
}

function fetch_user_by_id(PDO $pdo, int $userId): ?array
{
    if ($userId === 0) {
        return null;
    }
    $stmt = $pdo->prepare('SELECT * FROM bot_users WHERE id = :id LIMIT 1');
    $stmt->execute([':id' => $userId]);
    $row = $stmt->fetch();
    if ($row) {
        $row['is_blocked'] = (bool) ($row['is_blocked'] ?? false);
        $row['has_blocked_bot'] = (bool) ($row['has_blocked_bot'] ?? false);
    }
    return $row ?: null;
}

function fetch_user_by_telegram(PDO $pdo, int $telegramId): ?array
{
    $stmt = $pdo->prepare('SELECT * FROM bot_users WHERE telegram_id = :telegram_id LIMIT 1');
    $stmt->execute([':telegram_id' => $telegramId]);
    $row = $stmt->fetch();
    if ($row) {
        $row['is_blocked'] = (bool) ($row['is_blocked'] ?? false);
        $row['has_blocked_bot'] = (bool) ($row['has_blocked_bot'] ?? false);
    }
    return $row ?: null;
}

function fetch_user_by_username(PDO $pdo, string $username): ?array
{
    $stmt = $pdo->prepare('SELECT * FROM bot_users WHERE LOWER(username) = LOWER(:username) LIMIT 1');
    $stmt->execute([':username' => $username]);
    $row = $stmt->fetch();
    if ($row) {
        $row['is_blocked'] = (bool) ($row['is_blocked'] ?? false);
        $row['has_blocked_bot'] = (bool) ($row['has_blocked_bot'] ?? false);
    }
    return $row ?: null;
}

function set_user_block_status(PDO $pdo, int $userId, bool $blocked): void
{
    $stmt = $pdo->prepare('UPDATE bot_users SET is_blocked = :blocked WHERE id = :id');
    $stmt->execute([
        ':blocked' => $blocked ? 1 : 0,
        ':id' => $userId,
    ]);
}

function set_user_bot_block_status(PDO $pdo, int $userId, bool $blocked): void
{
    $stmt = $pdo->prepare('UPDATE bot_users SET has_blocked_bot = :blocked WHERE id = :id');
    $stmt->execute([
        ':blocked' => $blocked ? 1 : 0,
        ':id' => $userId,
    ]);
}

function format_user_display(array $user): string
{
    $name = trim((string) ($user['name'] ?? ''));
    $username = $user['username'] ?? '';
    $telegramId = $user['telegram_id'] ?? '';

    if ($username !== '') {
        $username = '@' . ltrim($username, '@');
    }

    $parts = array_filter([
        $name !== '' ? $name : null,
        $username !== '' ? $username : null,
    ]);

    $display = implode(' | ', $parts);
    if ($display === '') {
        $display = (string) $telegramId;
    }

    return $display;
}

function escape_html(string $text): string
{
    return htmlspecialchars($text, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
}

function format_datetime_for_display(?string $timestamp, array $config, string $format = 'Y-m-d H:i'): string
{
    if ($timestamp === null || $timestamp === '') {
        return '-';
    }

    $timezoneName = $config['app']['timezone'] ?? 'Asia/Tehran';

    try {
        $dateTime = new DateTime($timestamp, new DateTimeZone('UTC'));
    } catch (Exception $exception) {
        try {
            $dateTime = new DateTime($timestamp);
        } catch (Exception $fallbackException) {
            return $timestamp;
        }
    }

    try {
        $dateTime->setTimezone(new DateTimeZone($timezoneName));
    } catch (Exception $exception) {
        // اگر اعمال منطقه زمانی با خطا مواجه شد، همان مقدار را برمی‌گردانیم.
        return $dateTime->format($format);
    }

    return $dateTime->format($format);
}

function build_start_link(array $config, int $code): ?string
{
    $username = $config['telegram']['bot_username'] ?? '';
    $username = trim($username);
    if ($username === '') {
        return null;
    }

    $username = ltrim($username, '@');
    if ($username === '') {
        return null;
    }

    return sprintf('https://t.me/%s?start=%d', $username, $code);
}

function fetch_popular_overall(PDO $pdo): ?array
{
    try {
        $stmt = $pdo->query('SELECT code, download_count AS count FROM media_library WHERE position = 1 AND download_count > 0 ORDER BY download_count DESC LIMIT 1');
    } catch (Throwable $exception) {
        return null;
    }

    $row = $stmt->fetch();
    if (!$row) {
        return null;
    }

    $row['code'] = (int) $row['code'];
    $row['count'] = (int) $row['count'];
    return $row['count'] > 0 ? $row : null;
}

function fetch_popular_by_interval(PDO $pdo, int $days): ?array
{
    $days = max(1, (int) $days);
    $sql = "
        SELECT
            CAST(JSON_UNQUOTE(JSON_EXTRACT(details, '$.code')) AS UNSIGNED) AS code,
            COUNT(*) AS count
        FROM activity_logs
        WHERE action = 'user_download'
          AND created_at >= DATE_SUB(NOW(), INTERVAL {$days} DAY)
        GROUP BY code
        HAVING code IS NOT NULL AND code > 0
        ORDER BY count DESC
        LIMIT 1
    ";

    try {
        $stmt = $pdo->query($sql);
    } catch (Throwable $exception) {
        return null;
    }

    $row = $stmt->fetch();
    if (!$row) {
        return null;
    }

    $row['code'] = (int) $row['code'];
    $row['count'] = (int) $row['count'];
    return $row['count'] > 0 ? $row : null;
}

function format_popular_line(array $config, string $label, ?array $row): string
{
    if ($row === null) {
        return sprintf('%s: <i>داده‌ای نیست</i>', escape_html($label));
    }

    $code = (int) ($row['code'] ?? 0);
    $count = (int) ($row['count'] ?? 0);

    if ($code === 0 || $count === 0) {
        return sprintf('%s: <i>داده‌ای نیست</i>', escape_html($label));
    }

    $line = sprintf('%s: <b>کد %d</b> — <b>%d</b> دانلود', escape_html($label), $code, $count);
    $link = build_start_link($config, $code);
    if ($link) {
        $line .= sprintf(' — <a href="%s">لینک مستقیم</a>', escape_html($link));
    }

    return $line;
}

function send_user_stats_report(PDO $pdo, array $config, int $chatId, array $user): void
{
    $token = $config['telegram']['bot_token'];
    $actor = (string) ($user['telegram_id'] ?? '');

    $summaryStmt = $pdo->prepare("
        SELECT
            COUNT(*) AS total,
            COUNT(DISTINCT CAST(JSON_UNQUOTE(JSON_EXTRACT(details, '$.code')) AS UNSIGNED)) AS unique_codes,
            MAX(created_at) AS last_download
        FROM activity_logs
        WHERE action = 'user_download' AND actor = :actor
    ");
    $summaryStmt->execute([':actor' => $actor]);
    $summary = $summaryStmt->fetch() ?: ['total' => 0, 'unique_codes' => 0, 'last_download' => null];

    $codesStmt = $pdo->prepare("
        SELECT
            CAST(JSON_UNQUOTE(JSON_EXTRACT(details, '$.code')) AS UNSIGNED) AS code,
            COUNT(*) AS downloads,
            MAX(created_at) AS last_time
        FROM activity_logs
        WHERE action = 'user_download' AND actor = :actor
        GROUP BY code
        HAVING code IS NOT NULL AND code > 0
        ORDER BY last_time DESC
        LIMIT 10
    ");
    $codesStmt->execute([':actor' => $actor]);
    $codes = $codesStmt->fetchAll();

    $messageLines = [
        "گزارش کاربر 📊",
        format_user_display($user),
        "شناسه: {$user['telegram_id']}",
        "تاریخ عضویت: " . format_datetime_for_display($user['joined_at'] ?? null, $config),
        "آخرین فعالیت: " . format_datetime_for_display($user['last_seen_at'] ?? null, $config),
        $user['is_blocked'] ? "وضعیت: مسدود شده توسط ادمین ❌" : "وضعیت: مجاز ✅",
        !empty($user['has_blocked_bot']) ? "کاربر ربات را بلاک کرده است ❗️" : "کاربر ربات را بلاک نکرده ✅",
        "دانلود کل: " . ($summary['total'] ?? 0),
        "کدهای مختلف: " . ($summary['unique_codes'] ?? 0),
        "آخرین دانلود: " . format_datetime_for_display($summary['last_download'] ?? null, $config),
    ];

    if ($codes) {
        $messageLines[] = "";
        $messageLines[] = "کدهای دانلود شده:";
        foreach ($codes as $row) {
            $codeNumber = (int) $row['code'];
            $line = "- کد {$codeNumber} (تعداد: {$row['downloads']})";
            $link = build_start_link($config, $codeNumber);
            if ($link) {
                $line .= " → {$link}";
            }
            $messageLines[] = $line;
        }
    }

    telegram_send_message(
        $token,
        $chatId,
        implode("\n", $messageLines),
        [
            'reply_markup' => make_inline_keyboard([
                [[
                    'text' => 'بازگشت ↩️',
                    'callback_data' => 'admin_user_menu',
                ]],
            ]),
        ]
    );
}
function handle_admin_single_upload(PDO $pdo, array $config, int $chatId, array $mediaInfo): bool
{
    $token = $config['telegram']['bot_token'];
    $logFile = $config['app']['log_file'];

    try {
        $code = admin_next_code($pdo);

        $stmt = $pdo->prepare(
            'INSERT INTO media_library (code, media_type, file_id, caption, performer, title, position, created_by)
             VALUES (:code, :media_type, :file_id, :caption, :performer, :title, 1, :created_by)'
        );
        $stmt->execute([
            ':code' => $code,
            ':media_type' => $mediaInfo['type'],
            ':file_id' => $mediaInfo['file_id'],
            ':caption' => $mediaInfo['caption'],
            ':performer' => $mediaInfo['performer'],
            ':title' => $mediaInfo['title'],
            ':created_by' => $chatId,
        ]);

        $mediaId = (int) $pdo->lastInsertId();

        $messageId = forward_media_to_group($config, $mediaInfo, $code, 1);
        if ($messageId !== null) {
            $update = $pdo->prepare('UPDATE media_library SET group_message_id = :message_id WHERE id = :id');
            $update->execute([':message_id' => $messageId, ':id' => $mediaId]);
        }

        record_activity($pdo, (string) $chatId, 'admin_single_upload', [
            'code' => $code,
            'media_type' => $mediaInfo['type'],
        ]);

        reset_admin_state($pdo, $chatId);
        notify_admin_upload_success($config, $token, $chatId, $code, 1);
    } catch (Throwable $exception) {
        bot_log($logFile, 'single upload failed', ['error' => $exception->getMessage()]);
        telegram_send_message($token, $chatId, "اوپس! مشکل کوچیکی پیش اومد 😵 لطفاً دوباره امتحان کن یا بعداً خبرم کن.");
    }

    return true;
}

function handle_admin_batch_media(PDO $pdo, array $config, int $chatId, array $mediaInfo, array $state): bool
{
    $token = $config['telegram']['bot_token'];

    if (($state['pending_mode'] ?? 'idle') !== 'batch') {
        telegram_send_message($token, $chatId, "برای آپلود گروهی اول از منو گزینه «آپلود چندتایی» رو انتخاب کن.");
        return true;
    }

    $code = $state['pending_code'];
    $position = (int) ($state['pending_position'] ?? 1);

    if ($code === null) {
        $code = admin_next_code($pdo);
        $position = 1;
        set_admin_state($pdo, $chatId, [
            'pending_code' => $code,
            'pending_position' => $position,
        ]);
    }

    $stmt = $pdo->prepare(
        'INSERT INTO media_library (code, media_type, file_id, caption, performer, title, position, created_by)
         VALUES (:code, :media_type, :file_id, :caption, :performer, :title, :position, :created_by)'
    );
    $stmt->execute([
        ':code' => $code,
        ':media_type' => $mediaInfo['type'],
        ':file_id' => $mediaInfo['file_id'],
        ':caption' => $mediaInfo['caption'],
        ':performer' => $mediaInfo['performer'],
        ':title' => $mediaInfo['title'],
        ':position' => $position,
        ':created_by' => $chatId,
    ]);

    $mediaId = (int) $pdo->lastInsertId();
    $messageId = forward_media_to_group($config, $mediaInfo, $code, $position);
    if ($messageId !== null) {
        $update = $pdo->prepare('UPDATE media_library SET group_message_id = :message_id WHERE id = :id');
        $update->execute([':message_id' => $messageId, ':id' => $mediaId]);
    }

    set_admin_state($pdo, $chatId, [
        'pending_code' => $code,
        'pending_position' => $position + 1,
    ]);

    record_activity($pdo, (string) $chatId, 'admin_batch_append', [
        'code' => $code,
        'position' => $position,
        'media_type' => $mediaInfo['type'],
    ]);

    telegram_send_message(
        $token,
        $chatId,
        sprintf("%s ذخیره شد ✅ (کد %d - آیتم %d)", batch_media_title($mediaInfo), $code, $position),
        [
            'reply_markup' => make_inline_keyboard([
                [[
                    'text' => 'پایان آپلود ✅',
                    'callback_data' => 'admin_finish',
                ]],
                [[
                    'text' => 'بازگشت ↩️',
                    'callback_data' => 'admin_upload_menu',
                ]],
            ]),
        ]
    );

    return true;
}

function handle_admin_delete_code(PDO $pdo, array $config, int $chatId, string $text): void
{
    $token = $config['telegram']['bot_token'];
    $code = preg_replace('/\D+/', '', $text);

    if ($code === '' || !is_valid_code($code)) {
        telegram_send_message($token, $chatId, "کد معتبر نبود 😅 دوباره فقط عدد مربوط به رسانه رو بفرست.");
        return;
    }

    $stmt = $pdo->prepare('SELECT id, group_message_id FROM media_library WHERE code = :code');
    $stmt->execute([':code' => $code]);
    $items = $stmt->fetchAll();

    if ($items === []) {
        telegram_send_message($token, $chatId, "چیزی با این کد پیدا نکردم ❔ اگر مطمئنی وجود داره، دوباره امتحان کن یا از منو برگرد.");
        return;
    }

    $groupId = $config['telegram']['admin_group_id'] ?? 0;
    $token = $config['telegram']['bot_token'];

    foreach ($items as $item) {
        if ($groupId && !empty($item['group_message_id'])) {
            try {
                telegram_request($token, 'deleteMessage', [
                    'chat_id' => $groupId,
                    'message_id' => $item['group_message_id'],
                ]);
            } catch (Throwable $exception) {
                bot_log($config['app']['log_file'], 'delete group message failed', [
                    'code' => $code,
                    'message_id' => $item['group_message_id'],
                    'error' => $exception->getMessage(),
                ]);
            }
        }
    }

    $pdo->prepare('DELETE FROM media_library WHERE code = :code')->execute([':code' => $code]);
    record_activity($pdo, (string) $chatId, 'admin_delete_media', ['code' => $code, 'items' => count($items)]);

    telegram_send_message(
        $token,
        $chatId,
        "کد {$code} با موفقیت حذف شد 🗑️\nاگر کد دیگه‌ای هم هست همینجا بفرست یا با «بازگشت» برگرد.",
        [
            'reply_markup' => make_inline_keyboard([
                [[
                    'text' => 'بازگشت ↩️',
                    'callback_data' => 'admin_manage',
                ]],
            ]),
        ]
    );
}

function handle_admin_info_code(PDO $pdo, array $config, int $chatId, string $text): void
{
    $token = $config['telegram']['bot_token'];
    $code = preg_replace('/\D+/', '', $text);

    if ($code === '' || !is_valid_code($code)) {
        telegram_send_message($token, $chatId, "برای دریافت گزارش، فقط عدد کد رسانه رو بفرست 📊");
        return;
    }

    $stmt = $pdo->prepare(
        'SELECT code, media_type, caption, title, download_count, created_by, created_at, position
         FROM media_library
         WHERE code = :code
         ORDER BY position ASC'
    );
    $stmt->execute([':code' => $code]);
    $rows = $stmt->fetchAll();

    if ($rows === []) {
        telegram_send_message($token, $chatId, "رسانه‌ای با این کد نداریم ❔ اگر شک داری، از منو برگرد یا کد دیگه‌ای امتحان کن.");
        return;
    }

    $primary = $rows[0];
    $downloadCount = (int) $primary['download_count'];
    $createdBy = (int) $primary['created_by'];
    $createdAt = $primary['created_at'];

    $adminInfo = '';
    if ($createdBy !== 0) {
        $adminStmt = $pdo->prepare('SELECT username, full_name FROM admins WHERE telegram_id = :telegram_id LIMIT 1');
        $adminStmt->execute([':telegram_id' => $createdBy]);
        $adminRow = $adminStmt->fetch();
        if ($adminRow) {
            $name = $adminRow['full_name'] ?: $adminRow['username'] ?: (string) $createdBy;
            $adminInfo = $name;
        }
    }

    $uniqueUsers = $pdo->prepare("
        SELECT COUNT(DISTINCT actor) AS unique_users
        FROM activity_logs
        WHERE action = 'user_download'
          AND CAST(JSON_UNQUOTE(JSON_EXTRACT(details, '$.code')) AS UNSIGNED) = :code
    ");
    $uniqueUsers->execute([':code' => $code]);
    $uniqueCount = (int) $uniqueUsers->fetchColumn();

    $messageLines = [
        "گزارش رسانه 📈",
        "کد: {$code}",
        "تعداد آیتم‌ها: " . count($rows),
        "دانلودها: {$downloadCount}",
        "کاربران یکتا: {$uniqueCount}",
    ];

    if ($adminInfo !== '') {
        $messageLines[] = "آپلود توسط: {$adminInfo}";
    }

    if ($createdAt) {
        $messageLines[] = "تاریخ ایجاد: {$createdAt}";
    }

    $startLink = build_start_link($config, (int) $code);
    if ($startLink) {
        $messageLines[] = "لینک مستقیم: {$startLink}";
    }

    telegram_send_message(
        $token,
        $chatId,
        implode("\n", $messageLines),
        [
            'reply_markup' => make_inline_keyboard([
                [[
                    'text' => 'بازگشت ↩️',
                    'callback_data' => 'admin_manage',
                ]],
            ]),
        ]
    );
}

function finalize_batch_upload(PDO $pdo, array $config, int $chatId, array $state): void
{
    $token = $config['telegram']['bot_token'];

    if (($state['pending_mode'] ?? 'idle') !== 'batch' || empty($state['pending_code'])) {
        telegram_send_message($token, $chatId, "آپلود گروهی فعالی پیدا نکردم 🤔 اول از منو گزینه «آپلود چندتایی» رو انتخاب کن.");
        return;
    }

    $code = (int) $state['pending_code'];

    $stmt = $pdo->prepare('SELECT COUNT(*) FROM media_library WHERE code = :code');
    $stmt->execute([':code' => $code]);
    $count = (int) $stmt->fetchColumn();

    if ($count === 0) {
        telegram_send_message($token, $chatId, "ظاهراً هنوز فایلی با این کد ثبت نشده. چندتا فایل بفرست بعد دوباره پایان رو بزن.");
        return;
    }

    record_activity($pdo, (string) $chatId, 'admin_batch_upload', ['code' => $code, 'count' => $count]);

    reset_admin_state($pdo, $chatId);
    notify_admin_upload_success($config, $token, $chatId, $code, $count);
}

function notify_admin_upload_success(array $config, string $token, int $chatId, int $code, int $count): void
{
    $keyboard = make_inline_keyboard([
        [[
            'text' => 'آپلود جدید ➕',
            'callback_data' => 'admin_upload_menu',
        ]],
        [[
            'text' => 'مدیریت رسانه 🗂️',
            'callback_data' => 'admin_manage',
        ]],
        [[
            'text' => 'بازگشت ↩️',
            'callback_data' => 'admin_back',
        ]],
    ]);

    $message = sprintf("همه چی اوکی شد ✅\nکد اختصاصی: %d\nتعداد آیتم‌ها: %d", $code, $count);
    $startLink = build_start_link($config, $code);
    if ($startLink) {
        $message .= "\nلینک مستقیم: {$startLink}";
    }
    $message .= "\nهر وقت خواستی می‌تونی از منو ادامه بدی.";

    telegram_send_message(
        $token,
        $chatId,
        $message,
        ['reply_markup' => $keyboard]
    );
}

function forward_media_to_group(array $config, array $mediaInfo, int $code, int $position): ?int
{
    $groupId = $config['telegram']['admin_group_id'] ?? 0;
    if (!$groupId) {
        return null;
    }

    $token = $config['telegram']['bot_token'];
    $heading = "کد {$code}" . ($position > 1 ? " • آیتم {$position}" : '');
    $caption = $mediaInfo['caption'] !== '' ? $mediaInfo['caption'] . "\n" . $heading : $heading;
    $startLink = build_start_link($config, $code);
    if ($startLink) {
        $caption .= "\n🔗 {$startLink}";
    }

    switch ($mediaInfo['type']) {
        case 'audio':
            $result = telegram_request($token, 'sendAudio', [
                'chat_id' => $groupId,
                'audio' => $mediaInfo['file_id'],
                'caption' => $caption,
                'title' => $mediaInfo['title'],
                'performer' => $mediaInfo['performer'],
            ]);
            return $result['message_id'] ?? null;
        case 'video':
            $result = telegram_request($token, 'sendVideo', [
                'chat_id' => $groupId,
                'video' => $mediaInfo['file_id'],
                'caption' => $caption,
            ]);
            return $result['message_id'] ?? null;
        case 'photo':
            $result = telegram_request($token, 'sendPhoto', [
                'chat_id' => $groupId,
                'photo' => $mediaInfo['file_id'],
                'caption' => $caption,
            ]);
            return $result['message_id'] ?? null;
        default:
            $result = telegram_request($token, 'sendDocument', [
                'chat_id' => $groupId,
                'document' => $mediaInfo['file_id'],
                'caption' => $caption,
            ]);
            return $result['message_id'] ?? null;
    }
}

function batch_media_title(array $mediaInfo): string
{
    if ($mediaInfo['title'] !== '') {
        return $mediaInfo['title'];
    }

    if ($mediaInfo['caption'] !== '') {
        return $mediaInfo['caption'];
    }

    return ucfirst($mediaInfo['type']);
}

function admin_next_code(PDO $pdo): int
{
    $result = $pdo->query('SELECT COALESCE(MAX(code), 0) FROM media_library');
    $latest = (int) $result->fetchColumn();
    return $latest + 1;
}

function record_activity(PDO $pdo, string $actor, string $action, array $details): void
{
    $stmt = $pdo->prepare('INSERT INTO activity_logs (actor, action, details) VALUES (:actor, :action, :details)');
    $stmt->execute([
        ':actor' => $actor,
        ':action' => $action,
        ':details' => json_encode($details, JSON_UNESCAPED_UNICODE),
    ]);
}

function send_admin_stats(PDO $pdo, array $config, int $chatId): void
{
    $token = $config['telegram']['bot_token'];
 
    $stats = [];
 
    bot_log($config['app']['log_file'], 'admin_stats_requested', [
        'chat_id' => $chatId,
    ]);

    try {
        $stats['total_users'] = (int) $pdo->query('SELECT COUNT(*) FROM bot_users')->fetchColumn();
        $stats['blocked_users'] = (int) $pdo->query('SELECT COUNT(*) FROM bot_users WHERE is_blocked = 1')->fetchColumn();
        $stats['bot_blocked_users'] = (int) $pdo->query('SELECT COUNT(*) FROM bot_users WHERE has_blocked_bot = 1')->fetchColumn();
        $stats['active_users_total'] = (int) $pdo->query('SELECT COUNT(*) FROM bot_users WHERE is_blocked = 0 AND has_blocked_bot = 0')->fetchColumn();
        $stats['total_codes'] = (int) $pdo->query('SELECT COUNT(DISTINCT code) FROM media_library')->fetchColumn();
        $stats['daily_active'] = (int) $pdo->query("SELECT COUNT(*) FROM bot_users WHERE is_blocked = 0 AND has_blocked_bot = 0 AND last_seen_at >= DATE_SUB(UTC_TIMESTAMP(), INTERVAL 1 DAY)")->fetchColumn();
        $stats['weekly_active'] = (int) $pdo->query("SELECT COUNT(*) FROM bot_users WHERE is_blocked = 0 AND has_blocked_bot = 0 AND last_seen_at >= DATE_SUB(UTC_TIMESTAMP(), INTERVAL 7 DAY)")->fetchColumn();
        $stats['monthly_active'] = (int) $pdo->query("SELECT COUNT(*) FROM bot_users WHERE is_blocked = 0 AND has_blocked_bot = 0 AND last_seen_at >= DATE_SUB(UTC_TIMESTAMP(), INTERVAL 30 DAY)")->fetchColumn();
        $stats['total_downloads'] = (int) $pdo->query("SELECT COALESCE(SUM(download_count),0) FROM media_library")->fetchColumn();
        $stats['pending_broadcasts'] = (int) $pdo->query("SELECT COUNT(*) FROM broadcast_jobs WHERE status IN ('pending','running')")->fetchColumn();
        $stats['completed_broadcasts'] = (int) $pdo->query("SELECT COUNT(*) FROM broadcast_jobs WHERE status = 'completed'")->fetchColumn();
        $stats['failed_broadcasts'] = (int) $pdo->query("SELECT COUNT(*) FROM broadcast_jobs WHERE status = 'cancelled'")->fetchColumn();
    } catch (Throwable $exception) {
        bot_log($config['app']['log_file'], 'admin_stats_query_failed', [
            'chat_id' => $chatId,
            'error' => $exception->getMessage(),
        ]);
        telegram_send_message($token, $chatId, 'گرفتن آمار با خطا مواجه شد ❗️');
        return;
    }
 
    $popularOverall = fetch_popular_overall($pdo);
    $popularDaily = fetch_popular_by_interval($pdo, 1);
    $popularWeekly = fetch_popular_by_interval($pdo, 7);
    $popularMonthly = fetch_popular_by_interval($pdo, 30);
 
    $message = "📊 <b>آمار کلی</b>\n"
        . "👥 <b>کاربران کل:</b> {$stats['total_users']}\n"
        . "✅ <b>کاربران فعال:</b> {$stats['active_users_total']}\n"
        . "⛔️ <b>مسدود توسط ادمین:</b> {$stats['blocked_users']}\n"
        . "🚫 <b>کاربرانی که ربات را بلاک کرده‌اند:</b> {$stats['bot_blocked_users']}\n"
        . "🔢 <b>کدهای ثبت‌شده:</b> {$stats['total_codes']}\n"
        . "⬇️ <b>دانلودهای ثبت‌شده:</b> {$stats['total_downloads']}\n"
        . "📨 <b>ارسال‌های همگانی:</b> در صف {$stats['pending_broadcasts']} | تکمیل‌شده {$stats['completed_broadcasts']} | لغوشده {$stats['failed_broadcasts']}\n\n"
        . "🟢 <b>کاربران فعال اخیر</b>\n"
        . "• امروز: <b>{$stats['daily_active']}</b> کاربر\n"
        . "• هفتگی: <b>{$stats['weekly_active']}</b> کاربر\n"
        . "• ماهیانه: <b>{$stats['monthly_active']}</b> کاربر\n\n"
        . "🏆 <b>محبوب‌ترین کدها</b>\n"
        . format_popular_line($config, '• کلی', $popularOverall) . "\n"
        . format_popular_line($config, '• امروز', $popularDaily) . "\n"
        . format_popular_line($config, '• هفتگی', $popularWeekly) . "\n"
        . format_popular_line($config, '• ماهیانه', $popularMonthly);

    try {
        telegram_send_message($token, $chatId, $message, [
            'parse_mode' => 'HTML',
            'disable_web_page_preview' => true,
            'reply_markup' => make_inline_keyboard([
                [[
                    'text' => 'بازگشت ↩️',
                    'callback_data' => 'admin_menu',
                ]],
            ]),
        ]);
        bot_log($config['app']['log_file'], 'admin_stats_sent', [
            'chat_id' => $chatId,
        ]);
    } catch (Throwable $exception) {
        bot_log($config['app']['log_file'], 'admin_stats_failed', [
            'chat_id' => $chatId,
            'stats' => $stats,
            'error' => $exception->getMessage(),
        ]);
        telegram_send_message($token, $chatId, 'ارسال آمار با خطا مواجه شد ❗️');
    }
}

function ensure_admin_record(PDO $pdo, array $profile): void
{
    $telegramId = (int) ($profile['id'] ?? 0);
    if ($telegramId === 0) {
        return;
    }

    $username = clean_string($profile['username'] ?? '');
    $fullName = trim(sprintf(
        '%s %s',
        clean_string($profile['first_name'] ?? ''),
        clean_string($profile['last_name'] ?? '')
    ));

    $stmt = $pdo->prepare('SELECT telegram_id FROM admins WHERE telegram_id = :telegram_id LIMIT 1');
    $stmt->execute([':telegram_id' => $telegramId]);
    $exists = $stmt->fetchColumn();

    if ($exists) {
        $update = $pdo->prepare('UPDATE admins SET username = :username, full_name = :full_name WHERE telegram_id = :telegram_id');
        $update->execute([
            ':telegram_id' => $telegramId,
            ':username' => $username,
            ':full_name' => $fullName,
        ]);
    } else {
        $insert = $pdo->prepare('INSERT INTO admins (telegram_id, username, full_name) VALUES (:telegram_id, :username, :full_name)');
        $insert->execute([
            ':telegram_id' => $telegramId,
            ':username' => $username,
            ':full_name' => $fullName,
        ]);
    }
}

function get_admin_state(PDO $pdo, int $telegramId): array
{
    $stmt = $pdo->prepare('SELECT pending_mode, pending_code, pending_position, pending_group_id FROM admins WHERE telegram_id = :telegram_id LIMIT 1');
    $stmt->execute([':telegram_id' => $telegramId]);
    $row = $stmt->fetch();

    if (!$row) {
        return [
            'pending_mode' => 'idle',
            'pending_code' => null,
            'pending_position' => 1,
            'pending_group_id' => null,
        ];
    }

    return [
        'pending_mode' => $row['pending_mode'],
        'pending_code' => $row['pending_code'],
        'pending_position' => (int) $row['pending_position'],
        'pending_group_id' => $row['pending_group_id'],
    ];
}

function set_admin_state(PDO $pdo, int $telegramId, array $fields): array
{
    if ($fields === []) {
        return get_admin_state($pdo, $telegramId);
    }

    $setParts = [];
    $params = [':telegram_id' => $telegramId];
    foreach ($fields as $column => $value) {
        $setParts[] = sprintf('%s = :%s', $column, $column);
        $params[sprintf(':%s', $column)] = $value;
    }

    $sql = sprintf('UPDATE admins SET %s WHERE telegram_id = :telegram_id', implode(', ', $setParts));
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);

    return get_admin_state($pdo, $telegramId);
}

function reset_admin_state(PDO $pdo, int $telegramId): void
{
    set_admin_state($pdo, $telegramId, [
        'pending_mode' => 'idle',
        'pending_code' => null,
        'pending_position' => 1,
        'pending_group_id' => null,
    ]);
}

function extract_media_info(array $message): ?array
{
    $caption = clean_string($message['caption'] ?? '');
    $mediaGroupId = $message['media_group_id'] ?? null;

    if (isset($message['audio'])) {
        $audio = $message['audio'];
        return [
            'type' => 'audio',
            'file_id' => $audio['file_id'],
            'caption' => $caption,
            'performer' => clean_string($audio['performer'] ?? ''),
            'title' => clean_string($audio['title'] ?? ''),
            'media_group_id' => $mediaGroupId,
        ];
    }

    if (isset($message['video'])) {
        $video = $message['video'];
        return [
            'type' => 'video',
            'file_id' => $video['file_id'],
            'caption' => $caption,
            'performer' => '',
            'title' => clean_string($video['file_name'] ?? ''),
            'media_group_id' => $mediaGroupId,
        ];
    }

    if (isset($message['photo'])) {
        $photos = $message['photo'];
        $photo = end($photos);
        return [
            'type' => 'photo',
            'file_id' => $photo['file_id'],
            'caption' => $caption,
            'performer' => '',
            'title' => $caption,
            'media_group_id' => $mediaGroupId,
        ];
    }

    if (isset($message['document'])) {
        $doc = $message['document'];
        return [
            'type' => 'document',
            'file_id' => $doc['file_id'],
            'caption' => $caption,
            'performer' => '',
            'title' => clean_string($doc['file_name'] ?? ''),
            'media_group_id' => $mediaGroupId,
        ];
    }

    return null;
}

function get_telegram_bot_id(array $config): int
{
    static $botId = null;

    if ($botId !== null) {
        return $botId;
    }

    $configured = (int) ($config['telegram']['bot_id'] ?? 0);
    if ($configured > 0) {
        $botId = $configured;
        return $botId;
    }

    $token = $config['telegram']['bot_token'];
    try {
        $me = telegram_request($token, 'getMe');
        $botId = (int) ($me['id'] ?? 0);
    } catch (Throwable $exception) {
        bot_log($config['app']['log_file'], 'get_me_failed', ['error' => $exception->getMessage()]);
        $botId = 0;
    }

    return $botId;
}

function fetch_force_channels(PDO $pdo, bool $onlyActive = true): array
{
    $sql = 'SELECT * FROM force_channels';
    if ($onlyActive) {
        $sql .= ' WHERE is_active = 1';
    }
    $sql .= ' ORDER BY created_at DESC';

    $stmt = $pdo->query($sql);
    return $stmt ? $stmt->fetchAll() : [];
}

function fetch_force_channel(PDO $pdo, int $id): ?array
{
    $stmt = $pdo->prepare('SELECT * FROM force_channels WHERE id = :id LIMIT 1');
    $stmt->execute([':id' => $id]);
    $row = $stmt->fetch();
    return $row ?: null;
}

function fetch_force_channel_by_chat(PDO $pdo, int $chatId): ?array
{
    $stmt = $pdo->prepare('SELECT * FROM force_channels WHERE chat_id = :chat_id LIMIT 1');
    $stmt->execute([':chat_id' => $chatId]);
    $row = $stmt->fetch();
    return $row ?: null;
}

function save_force_channel(PDO $pdo, array $channel, int $addedBy): array
{
    $chatId = (int) ($channel['chat_id'] ?? 0);
    if ($chatId === 0) {
        throw new InvalidArgumentException('chat_id missing for force channel');
    }

    $title = clean_string($channel['title'] ?? '');
    $username = clean_string($channel['username'] ?? '');
    if ($username !== '') {
        $username = ltrim($username, '@');
    }
    $inviteLink = $channel['invite_link'] ?? null;
    $chatType = strtolower($channel['chat_type'] ?? 'channel');
    if (!in_array($chatType, ['channel', 'supergroup', 'bot'], true)) {
        $chatType = 'channel';
    }

    if ($chatType === 'bot') {
        if ($title === '') {
            $title = '@' . $username;
        }
        if ($inviteLink === null || $inviteLink === '') {
            $inviteLink = $username !== '' ? sprintf('https://t.me/%s', $username) : null;
        }
    }

    $existing = fetch_force_channel_by_chat($pdo, $chatId);
    if ($existing) {
        $updateSql = 'UPDATE force_channels SET title = :title, username = :username, invite_link = :invite_link, chat_type = :chat_type, is_active = 1, added_by = :added_by';
        if ($chatType === 'bot') {
            $updateSql .= ', bot_is_admin = 1, target_joins = NULL, current_joins = 0, total_joins = 0, total_left = 0, total_invites = 0, last_checked_at = NOW()';
        } else {
            $updateSql .= ', bot_is_admin = 0, target_joins = NULL, current_joins = 0, total_joins = 0, total_left = 0, last_checked_at = NULL';
        }
        $updateSql .= ' WHERE chat_id = :chat_id';

        $stmt = $pdo->prepare($updateSql);
        $stmt->execute([
            ':title' => $title,
            ':username' => $username !== '' ? $username : null,
            ':invite_link' => $inviteLink,
            ':chat_type' => $chatType,
            ':added_by' => $addedBy,
            ':chat_id' => $chatId,
        ]);
        return fetch_force_channel($pdo, (int) $existing['id']);
    }

    $stmt = $pdo->prepare('INSERT INTO force_channels (chat_id, chat_type, title, username, invite_link, added_by, bot_is_admin) VALUES (:chat_id, :chat_type, :title, :username, :invite_link, :added_by, :bot_is_admin)');
    $stmt->execute([
        ':chat_id' => $chatId,
        ':chat_type' => $chatType,
        ':title' => $title,
        ':username' => $username !== '' ? $username : null,
        ':invite_link' => $inviteLink,
        ':added_by' => $addedBy,
        ':bot_is_admin' => $chatType === 'bot' ? 1 : 0,
    ]);

    $id = (int) $pdo->lastInsertId();
    return fetch_force_channel($pdo, $id) ?? [];
}

function update_force_channel_invite_link(PDO $pdo, int $channelId, ?string $inviteLink): void
{
    $stmt = $pdo->prepare('UPDATE force_channels SET invite_link = :invite_link WHERE id = :id');
    $stmt->execute([
        ':invite_link' => $inviteLink,
        ':id' => $channelId,
    ]);
}

function update_force_channel_admin_status(PDO $pdo, int $channelId, bool $isAdmin): void
{
    $stmt = $pdo->prepare('UPDATE force_channels SET bot_is_admin = :admin, last_checked_at = NOW() WHERE id = :id');
    $stmt->execute([
        ':admin' => $isAdmin ? 1 : 0,
        ':id' => $channelId,
    ]);
}

function deactivate_force_channel(PDO $pdo, int $channelId): void
{
    $stmt = $pdo->prepare('UPDATE force_channels SET is_active = 0 WHERE id = :id');
    $stmt->execute([':id' => $channelId]);
}

function toggle_force_channel_status(PDO $pdo, int $channelId, bool $active): void
{
    $stmt = $pdo->prepare('UPDATE force_channels SET is_active = :active WHERE id = :id');
    $stmt->execute([
        ':active' => $active ? 1 : 0,
        ':id' => $channelId,
    ]);
}

function looks_like_bot_username(string $value): bool
{
    $normalized = strtolower(trim($value));
    if ($normalized === '') {
        return false;
    }
    if ($normalized[0] === '@') {
        $normalized = substr($normalized, 1);
    }
    return $normalized !== '' && preg_match('/bot$/', $normalized) === 1;
}

function make_virtual_bot_chat_id(string $username): int
{
    $normalized = strtolower(ltrim($username, '@'));
    if ($normalized === '') {
        return -1;
    }
    $hash = intval(substr(hash('sha1', $normalized), 0, 15), 16);
    if ($hash <= 0) {
        $hash = 1;
    }
    $hash = $hash % 900000000000000000;
    if ($hash <= 0) {
        $hash = 1;
    }
    return -1 * $hash;
}

function is_force_channel_bot(array $channel): bool
{
    return strtolower((string) ($channel['chat_type'] ?? 'channel')) === 'bot';
}

function extract_force_channel_link(string $text): ?string
{
    if ($text === '') {
        return null;
    }

    if (preg_match('#https?://t\.me/[^\s]+#iu', $text, $match)) {
        return $match[0];
    }

    if (preg_match('#t\.me/[^\s]+#iu', $text, $match)) {
        return 'https://' . $match[0];
    }

    return null;
}

function resolve_force_channel_reference(array $config, array $message, string $text): ?array
{
    $token = $config['telegram']['bot_token'];

    if (isset($message['forward_from_chat']) && ($message['forward_from_chat']['type'] ?? '') === 'channel') {
        $chat = $message['forward_from_chat'];
        return [
            'chat_id' => (int) $chat['id'],
            'title' => clean_string($chat['title'] ?? ''),
            'username' => clean_string($chat['username'] ?? ''),
            'invite_link' => extract_force_channel_link($text),
            'chat_type' => $chat['type'] ?? 'channel',
        ];
    }

    if (isset($message['forward_from_chat']) && ($message['forward_from_chat']['type'] ?? '') === 'supergroup') {
        $chat = $message['forward_from_chat'];
        return [
            'chat_id' => (int) $chat['id'],
            'title' => clean_string($chat['title'] ?? ''),
            'username' => clean_string($chat['username'] ?? ''),
            'invite_link' => extract_force_channel_link($text),
            'chat_type' => $chat['type'] ?? 'supergroup',
        ];
    }

    $candidate = trim($text);
    if ($candidate === '') {
        return null;
    }

    $inviteLink = extract_force_channel_link($candidate);

    if ($inviteLink !== null && !preg_match('/@([A-Za-z0-9_]{5,32})/u', $candidate)) {
        // Private invite link without username; need forwarded post to obtain chat_id
        return ['error' => 'invite_only'];
    }

    if (preg_match('/@([A-Za-z0-9_]{5,32})/u', $candidate, $match)) {
        $candidate = '@' . $match[1];
    } elseif (preg_match('#t\.me/([A-Za-z0-9_]{5,32})#iu', $candidate, $match)) {
        $candidate = '@' . $match[1];
    }

    if (looks_like_bot_username($candidate)) {
        $username = ltrim($candidate, '@');
        $virtualId = make_virtual_bot_chat_id($username);
        $displayTitle = '@' . $username;
        return [
            'chat_id' => $virtualId,
            'chat_type' => 'bot',
            'title' => $displayTitle,
            'username' => $username,
            'invite_link' => sprintf('https://t.me/%s', $username),
        ];
    }

    try {
        $chat = telegram_request($token, 'getChat', ['chat_id' => $candidate]);

        $type = $chat['type'] ?? '';
        if ($type === 'private' && ($chat['is_bot'] ?? false)) {
            $username = clean_string($chat['username'] ?? ltrim($candidate, '@'));
            if ($username === '') {
                return ['error' => 'bot', 'username' => clean_string($candidate)];
            }
            return [
                'chat_id' => make_virtual_bot_chat_id($username),
                'chat_type' => 'bot',
                'title' => '@' . $username,
                'username' => $username,
                'invite_link' => sprintf('https://t.me/%s', $username),
            ];
        }

        if (!in_array($type, ['channel', 'supergroup'], true)) {
            return ['error' => 'unsupported', 'type' => $type];
        }

        return [
            'chat_id' => (int) ($chat['id'] ?? 0),
            'chat_type' => $type,
            'title' => clean_string($chat['title'] ?? ''),
            'username' => clean_string($chat['username'] ?? ''),
            'invite_link' => $inviteLink,
        ];
    } catch (Throwable $exception) {
        bot_log($config['app']['log_file'], 'force_channel_resolve_failed', [
            'candidate' => $candidate,
            'error' => $exception->getMessage(),
        ]);
    }

    return null;
}

function resolve_force_channel_link(array $channel): ?string
{
    if (!empty($channel['invite_link'])) {
        return $channel['invite_link'];
    }

    if (!empty($channel['username'])) {
        return sprintf('https://t.me/%s', ltrim((string) $channel['username'], '@'));
    }

    return null;
}

function verify_bot_admin_in_channel(array $config, array $channel): bool
{
    if (is_force_channel_bot($channel)) {
        return true;
    }
    $token = $config['telegram']['bot_token'];
    $botId = get_telegram_bot_id($config);
    if ($botId === 0) {
        return false;
    }

    $chatId = (int) ($channel['chat_id'] ?? 0);
    if ($chatId === 0) {
        return false;
    }

    try {
        $member = telegram_request($token, 'getChatMember', [
            'chat_id' => $chatId,
            'user_id' => $botId,
        ]);

        $status = $member['status'] ?? '';
        return in_array($status, ['administrator', 'creator'], true);
    } catch (Throwable $exception) {
        bot_log($config['app']['log_file'], 'verify_bot_admin_failed', [
            'chat_id' => $chatId,
            'error' => $exception->getMessage(),
        ]);
        return false;
    }
}

function check_user_membership(array $config, array $channel, int $userId): bool
{
    if (is_force_channel_bot($channel)) {
        return true;
    }
    $token = $config['telegram']['bot_token'];
    $chatId = (int) ($channel['chat_id'] ?? 0);
    if ($chatId === 0) {
        return false;
    }

    try {
        $member = telegram_request($token, 'getChatMember', [
            'chat_id' => $chatId,
            'user_id' => $userId,
        ]);
        $status = $member['status'] ?? '';
        return in_array($status, ['creator', 'administrator', 'member'], true);
    } catch (Throwable $exception) {
        bot_log($config['app']['log_file'], 'check_user_membership_failed', [
            'chat_id' => $chatId,
            'user_id' => $userId,
            'error' => $exception->getMessage(),
        ]);
        return false;
    }
}

function enforce_force_channels(PDO $pdo, array $config, array $chat, array $botUser): bool
{
    $channels = fetch_force_channels($pdo, true);
    if ($channels === []) {
        return true;
    }

    $chatId = (int) ($chat['id'] ?? 0);
    if ($chatId === 0) {
        return false;
    }

    $botUserId = (int) ($botUser['id'] ?? 0);
    if ($botUserId === 0) {
        return false;
    }

    $missing = [];

    foreach ($channels as $channel) {
        if (is_force_channel_bot($channel)) {
            continue;
        }
        if (!($channel['bot_is_admin'] ?? 0)) {
            continue;
        }

        if (!check_user_membership($config, $channel, $chatId)) {
            $missing[] = $channel;
            continue;
        }

        register_force_channel_join($pdo, $config, $channel, $botUser);
        try {
            $updatedChannel = fetch_force_channel($pdo, (int) ($channel['id'] ?? 0));
            if ($updatedChannel) {
                finalize_force_channel_if_completed($pdo, $config, $updatedChannel);
            }
        } catch (Throwable $exception) {
            bot_log($config['app']['log_file'], 'force_check_finalize_failed', [
                'channel_id' => $channel['id'] ?? null,
                'error' => $exception->getMessage(),
            ]);
        }
    }

    if ($missing === []) {
        return true;
    }

    $token = $config['telegram']['bot_token'];
    $buttons = [];
    foreach ($missing as $channel) {
        $link = resolve_force_channel_link($channel);
        $title = clean_string($channel['title'] ?? 'کانال');
        if ($link) {
            $buttons[] = [[
                'text' => $title,
                'url' => $link,
            ]];
        }
    }
    $buttons[] = [[
        'text' => 'عضو شدم ✅',
        'callback_data' => 'force_check',
    ]];

    $titles = array_map(static function (array $channel): string {
        $title = clean_string($channel['title'] ?? 'کانال');
        return sprintf('• %s', $title);
    }, $missing);
    $message = "برای استفاده از ربات باید عضو کانال‌های اسپانسر باشی 💡\n" . implode("\n", $titles) . "\n\nبعد از عضویت روی «عضو شدم ✅» بزن.";

    telegram_send_message($token, $chatId, $message, [
        'reply_markup' => make_inline_keyboard($buttons),
        'disable_web_page_preview' => true,
    ]);

    return false;
}

function refresh_force_channel_stats(PDO $pdo, array $config): void
{
    $channels = fetch_force_channels($pdo, true);
    foreach ($channels as $channel) {
        if ((int) ($channel['is_active'] ?? 0) !== 1) {
            continue;
        }

        try {
            $channel = synchronize_force_channel_counters($pdo, $config, $channel);
        } catch (Throwable $exception) {
            bot_log($config['app']['log_file'], 'force_refresh_sync_failed', [
                'channel_id' => $channel['id'] ?? null,
                'error' => $exception->getMessage(),
            ]);
            continue;
        }

        try {
            finalize_force_channel_if_completed($pdo, $config, $channel);
        } catch (Throwable $exception) {
            bot_log($config['app']['log_file'], 'force_refresh_finalize_failed', [
                'channel_id' => $channel['id'] ?? null,
                'error' => $exception->getMessage(),
            ]);
        }
    }
}

function process_user_callback(PDO $pdo, array $config, array $callback): bool
{
    $data = $callback['data'] ?? '';

    if ($data === 'force_check') {
        $from = $callback['from'] ?? [];
        $chat = [
            'id' => $from['id'] ?? ($callback['message']['chat']['id'] ?? 0),
            'username' => $from['username'] ?? '',
            'first_name' => $from['first_name'] ?? '',
            'last_name' => $from['last_name'] ?? '',
        ];

        $botUser = upsert_bot_user($pdo, $chat);
        if (!enforce_force_channels($pdo, $config, $chat, $botUser)) {
            return true;
        }

        refresh_force_channel_stats($pdo, $config);

        send_welcome($config['telegram']['bot_token'], $chat, $config);
        return true;
    }

    return false;
}

function send_force_channels_menu(PDO $pdo, array $config, int $chatId): void
{
    $allChannels = fetch_force_channels($pdo, false);
    $activeChannels = array_values(array_filter($allChannels, fn($channel) => (int) ($channel['is_active'] ?? 0) === 1));

    $lines = [
        "اینجا کانال‌های اسپانسر رو مدیریت می‌کنیم 📣",
    ];

    if ($activeChannels === []) {
        $lines[] = 'برای شروع از دکمه «افزودن کانال» استفاده کن.';
    } else {
        $lines[] = 'برای مدیریت هر کانال روی نامش لمس کن.';
    }

    $buttons = [];
    foreach ($activeChannels as $channel) {
        $title = clean_string($channel['title'] ?? 'کانال');
        if (is_force_channel_bot($channel)) {
            $username = $channel['username'] ?? '';
            $title = $username !== '' ? '@' . ltrim($username, '@') : $title;
            $title = '🤖 ' . $title;
        }
        if (mb_strlen($title) > 30) {
            $title = mb_substr($title, 0, 27) . '…';
        }

        $buttons[] = [[
            'text' => $title,
            'callback_data' => sprintf('force_show:%d', $channel['id']),
        ]];
    }

    $buttons[] = [[
        'text' => '➕ افزودن کانال',
        'callback_data' => 'force_add_start',
    ]];

    $buttons[] = [[
        'text' => 'بازگشت ↩️',
        'callback_data' => 'admin_menu',
    ]];

    telegram_send_message(
        $config['telegram']['bot_token'],
        $chatId,
        implode("\n", $lines),
        ['reply_markup' => make_inline_keyboard($buttons)]
    );
}

function build_force_channel_card_payload(array $config, array $channel, string $statusMessage = ''): array
{
    $channelId = (int) ($channel['id'] ?? 0);
    $title = clean_string($channel['title'] ?? 'کانال');
    $username = $channel['username'] ?? null;
    $inviteLink = resolve_force_channel_link($channel);
    $isActive = (int) ($channel['is_active'] ?? 0) === 1;
    $botIsAdmin = (int) ($channel['bot_is_admin'] ?? 0) === 1;
    $target = (int) ($channel['target_joins'] ?? 0);
    $current = (int) ($channel['current_joins'] ?? 0);
    $totalJoined = (int) ($channel['total_joins'] ?? $current);
    $totalLeft = (int) ($channel['total_left'] ?? 0);
    $chatType = strtolower((string) ($channel['chat_type'] ?? 'channel'));

    $lines = [];
    if ($statusMessage !== '') {
        $lines[] = $statusMessage;
        $lines[] = '';
    }

    if ($chatType === 'bot') {
        $lines[] = 'ربات تبلیغاتی ثبت شده ✅';
        $lines[] = sprintf('عنوان: %s', $title);
        if ($username) {
            $lines[] = sprintf('یوزرنیم: @%s', ltrim($username, '@'));
        }
        if ($inviteLink) {
            $lines[] = sprintf('لینک شروع: %s', $inviteLink);
        }

        $keyboardButtons = [
            [[
                'text' => 'حذف از لیست 🗑️',
                'callback_data' => sprintf('force_delete:%d', $channelId),
            ]],
            [[
                'text' => 'بازگشت ↩️',
                'callback_data' => 'admin_force_menu',
            ]],
        ];

        return [
            'text' => implode("\n", $lines),
            'reply_markup' => make_inline_keyboard($keyboardButtons),
        ];
    }

    // Stage 1: Bot is not yet admin – show guidance and single verify button.
    if (!$botIsAdmin) {
        $lines[] = 'کانال ذخیره شد ✅';
        $lines[] = '';
        $lines[] = 'لطفاً همین الان ربات را به عنوان ادمین کانال اضافه کن تا بتواند اعضا را بررسی کند.';
        if ($username) {
            $lines[] = sprintf('آیدی کانال: @%s', ltrim($username, '@'));
        }
        if ($inviteLink) {
            $lines[] = sprintf('لینک کانال: %s', $inviteLink);
        }
        $lines[] = 'بعد از اینکه ربات را ادمین کردی روی «ادمین شد ✅» بزن.';

        $keyboardButtons = [[[
            'text' => 'ادمین شد ✅',
            'callback_data' => sprintf('force_verify:%d', $channelId),
        ]]];

        return [
            'text' => implode("\n", $lines),
            'reply_markup' => make_inline_keyboard($keyboardButtons),
        ];
    }

    // Stage 2: Bot is admin but target not set yet – prompt for target.
    if ($target <= 0) {
        $lines[] = 'ربات الان ادمین کانال است ✅';
        $lines[] = '';
        $lines[] = 'حالا تعداد اعضایی که می‌خوای این کمپین جذب کنه رو مشخص کن.';
        $lines[] = 'روی «ثبت هدف 🎯» بزن و عدد هدف (مثلاً 2000) را بفرست.';

        $keyboardButtons = [[[
            'text' => 'ثبت هدف 🎯',
            'callback_data' => sprintf('force_edit_target:%d', $channelId),
        ]]];

        return [
            'text' => implode("\n", $lines),
            'reply_markup' => make_inline_keyboard($keyboardButtons),
        ];
    }

    // Stage 3: Full management card once target exists.
    $lines[] = 'جزئیات کانال باز شد ✅';
    $lines[] = sprintf('عنوان: %s', $title);
    if ($username) {
        $lines[] = sprintf('آیدی: @%s', ltrim($username, '@'));
    }
    if ($inviteLink) {
        $lines[] = sprintf('لینک: %s', $inviteLink);
    }
    $lines[] = $isActive ? 'وضعیت: فعال ✅' : 'وضعیت: غیرفعال ⛔️';
    $lines[] = sprintf('هدف کمپین: %d نفر', $target);

    $keyboardButtons = [];

    $keyboardButtons[] = [[
        'text' => sprintf('👥 عضو: %d', $current),
        'callback_data' => sprintf('force_stat:%d', $channelId),
    ], [
        'text' => sprintf('✅ جمع: %d', $totalJoined),
        'callback_data' => sprintf('force_stat:%d', $channelId),
    ], [
        'text' => sprintf('🚪 لفت: %d', $totalLeft),
        'callback_data' => sprintf('force_stat:%d', $channelId),
    ]];

    $keyboardButtons[] = [[
        'text' => 'ویرایش هدف 🎯',
        'callback_data' => sprintf('force_edit_target:%d', $channelId),
    ]];

    $keyboardButtons[] = [[
        'text' => 'به‌روزرسانی آمار 🔄',
        'callback_data' => sprintf('force_refresh:%d', $channelId),
    ]];

    $keyboardButtons[] = [[
        'text' => $isActive ? 'غیرفعال کردن ❌' : 'فعال کردن ✅',
        'callback_data' => sprintf('force_toggle:%d:%d', $channelId, $isActive ? 0 : 1),
    ]];

    $keyboardButtons[] = [[
        'text' => 'حذف از لیست 🗑️',
        'callback_data' => sprintf('force_delete:%d', $channelId),
    ]];

    $keyboardButtons[] = [[
        'text' => 'بازگشت ↩️',
        'callback_data' => 'admin_force_menu',
    ]];

    return [
        'text' => implode("\n", $lines),
        'reply_markup' => make_inline_keyboard($keyboardButtons),
    ];
}

function send_force_channel_card(array $config, int $chatId, array $channel, string $statusMessage = ''): void
{
    $payload = build_force_channel_card_payload($config, $channel, $statusMessage);
    telegram_send_message(
        $config['telegram']['bot_token'],
        $chatId,
        $payload['text'],
        ['reply_markup' => $payload['reply_markup']]
    );
}

function edit_force_channel_card(array $config, int $chatId, int $messageId, array $channel, string $statusMessage = ''): void
{
    $payload = build_force_channel_card_payload($config, $channel, $statusMessage);
    telegram_request($config['telegram']['bot_token'], 'editMessageText', [
        'chat_id' => $chatId,
        'message_id' => $messageId,
        'text' => $payload['text'],
        'reply_markup' => $payload['reply_markup'],
        'disable_web_page_preview' => true,
    ]);
}

function fetch_force_channel_by_invite(PDO $pdo, string $inviteLink): ?array
{
    $stmt = $pdo->prepare('SELECT * FROM force_channels WHERE invite_link = :invite LIMIT 1');
    $stmt->execute([':invite' => $inviteLink]);
    $row = $stmt->fetch();
    return $row ?: null;
}

function send_force_goal_prompt(array $config, int $chatId, int $channelId, bool $isUpdate = false): void
{
    $text = $isUpdate
        ? "تعداد جدید اعضایی که می‌خوای این کمپین جذب کنه را بنویس (مثلاً 2000)."
        : "تعداد اعضایی که می‌خوای با این کمپین جذب بشن را بنویس (مثلاً 2000).";
    telegram_send_message($config['telegram']['bot_token'], $chatId, $text);
}

function create_force_channel_invite_link(PDO $pdo, array $config, array $channel, int $target): ?string
{
    if (is_force_channel_bot($channel)) {
        return null;
    }
    $token = $config['telegram']['bot_token'];
    $chatId = (int) ($channel['chat_id'] ?? 0);
    $channelId = (int) ($channel['id'] ?? 0);

    if ($chatId === 0 || $channelId === 0) {
        return null;
    }

    $params = [
        'chat_id' => $chatId,
        'name' => sprintf('sponsor-%d-%d', $channelId, time()),
    ];

    $memberLimit = max(1, min($target, 99999));
    if ($target > 0) {
        $params['member_limit'] = $memberLimit;
    }

    try {
        $invite = telegram_request($token, 'createChatInviteLink', $params);
    } catch (Throwable $exception) {
        bot_log($config['app']['log_file'], 'create_invite_failed', [
            'chat_id' => $chatId,
            'error' => $exception->getMessage(),
            'params' => $params,
        ]);

        if (isset($params['member_limit'])) {
            unset($params['member_limit']);
            try {
                $invite = telegram_request($token, 'createChatInviteLink', $params);
            } catch (Throwable $fallbackException) {
                bot_log($config['app']['log_file'], 'create_invite_failed_fallback', [
                    'chat_id' => $chatId,
                    'error' => $fallbackException->getMessage(),
                    'params' => $params,
                ]);
                return null;
            }
        } else {
            return null;
        }
    }

    $inviteLink = $invite['invite_link'] ?? null;
    if (!$inviteLink) {
        return null;
    }

    $baselineCount = fetch_force_channel_member_count($config, $channel);

    if (force_channel_supports_progress($pdo)) {
        $storedTarget = $target > 0 ? $memberLimit : 0;
        update_force_channel_target($pdo, $channelId, $storedTarget, $inviteLink, $baselineCount);
    } else {
        update_force_channel_invite_link($pdo, $channelId, $inviteLink);
    }

    bot_log($config['app']['log_file'], 'force_invite_created', [
        'channel_id' => $channelId,
        'chat_id' => $chatId,
        'invite_link' => $inviteLink,
        'member_limit' => $target > 0 ? $memberLimit : null,
    ]);

    notify_force_channel_invite_created($config, $channel, $inviteLink, $target > 0 ? $memberLimit : null);

    return $inviteLink;
}

function register_force_channel_join(PDO $pdo, array $config, array $channel, array $botUser): void
{
    if (is_force_channel_bot($channel)) {
        return;
    }
    ensure_force_channel_member_structure($pdo);
    $channelId = (int) ($channel['id'] ?? 0);
    $telegramId = (int) ($botUser['telegram_id'] ?? 0);
    $userId = (int) ($botUser['id'] ?? 0);
    if ($channelId === 0 || $telegramId === 0 || $userId === 0) {
        return;
    }

    $supportsMembers = force_channel_supports_progress($pdo);

    $inserted = false;
    if ($supportsMembers) {
        try {
            $stmt = $pdo->prepare('INSERT IGNORE INTO force_channel_members (channel_id, user_id, telegram_id) VALUES (:channel_id, :user_id, :telegram_id)');
            $stmt->execute([
                ':channel_id' => $channelId,
                ':user_id' => $userId,
                ':telegram_id' => $telegramId,
            ]);
            $inserted = $stmt->rowCount() > 0;
        } catch (Throwable $exception) {
            bot_log($config['app']['log_file'], 'force_member_insert_failed', [
                'channel_id' => $channelId,
                'user_id' => $userId,
                'error' => $exception->getMessage(),
            ]);
        }
    }

    if (!$supportsMembers) {
        try {
            $pdo->prepare('UPDATE force_channels SET total_joins = total_joins + 1 WHERE id = :id')->execute([
                ':id' => $channelId,
            ]);
        } catch (Throwable $exception) {
            bot_log($config['app']['log_file'], 'force_member_total_update_failed', [
                'channel_id' => $channelId,
                'error' => $exception->getMessage(),
            ]);
        }
    } elseif ($inserted) {
        try {
            $pdo->prepare('UPDATE force_channels SET total_joins = total_joins + 1, total_invites = total_invites + 1 WHERE id = :id')->execute([
                ':id' => $channelId,
            ]);
        } catch (Throwable $exception) {
            bot_log($config['app']['log_file'], 'force_member_total_update_failed', [
                'channel_id' => $channelId,
                'error' => $exception->getMessage(),
            ]);
        }
    }

    if ($supportsMembers) {
        try {
            $pdo->prepare('UPDATE force_channels SET current_joins = (SELECT COUNT(*) FROM force_channel_members WHERE channel_id = :channel_id) WHERE id = :channel_id')->execute([
                ':channel_id' => $channelId,
            ]);
        } catch (Throwable $exception) {
            bot_log($config['app']['log_file'], 'force_member_current_update_failed', [
                'channel_id' => $channelId,
                'error' => $exception->getMessage(),
            ]);
        }
    }

    $updated = fetch_force_channel($pdo, $channelId);
    if (!$updated) {
        return;
    }

    if ((int) ($updated['target_joins'] ?? 0) > 0 && (int) ($updated['current_joins'] ?? 0) >= (int) ($updated['target_joins'] ?? 0)) {
        notify_force_channel_completed($config, $updated);
        delete_force_channel($pdo, $config, $channelId);
    }
}

function register_force_channel_leave(PDO $pdo, array $config, array $channel, array $userInfo): void
{
    if (is_force_channel_bot($channel)) {
        return;
    }
    ensure_force_channel_member_structure($pdo);
    $channelId = (int) ($channel['id'] ?? 0);
    $telegramId = (int) ($userInfo['id'] ?? 0);
    if ($channelId === 0 || $telegramId === 0) {
        return;
    }

    $supportsMembers = force_channel_supports_progress($pdo);
    $deleted = false;

    if ($supportsMembers) {
        try {
            $stmt = $pdo->prepare('DELETE FROM force_channel_members WHERE channel_id = :channel_id AND telegram_id = :telegram_id');
            $stmt->execute([
                ':channel_id' => $channelId,
                ':telegram_id' => $telegramId,
            ]);
            $deleted = $stmt->rowCount() > 0;
        } catch (Throwable $exception) {
            bot_log($config['app']['log_file'], 'force_member_delete_failed', [
                'channel_id' => $channelId,
                'user' => $telegramId,
                'error' => $exception->getMessage(),
            ]);
        }
    }

    if (!$supportsMembers) {
        try {
            $pdo->prepare('UPDATE force_channels SET total_left = total_left + 1 WHERE id = :id')->execute([
                ':id' => $channelId,
            ]);
        } catch (Throwable $exception) {
            bot_log($config['app']['log_file'], 'force_member_leave_update_failed', [
                'channel_id' => $channelId,
                'error' => $exception->getMessage(),
            ]);
        }
        return;
    }

    if (!$deleted) {
        return;
    }

    try {
        $pdo->prepare('UPDATE force_channels SET total_left = total_left + 1 WHERE id = :id')->execute([
            ':id' => $channelId,
        ]);
        $pdo->prepare('UPDATE force_channels SET current_joins = (SELECT COUNT(*) FROM force_channel_members WHERE channel_id = :channel_id) WHERE id = :channel_id')->execute([
            ':channel_id' => $channelId,
        ]);
    } catch (Throwable $exception) {
        bot_log($config['app']['log_file'], 'force_member_leave_update_failed', [
            'channel_id' => $channelId,
            'error' => $exception->getMessage(),
        ]);
    }
}

function notify_force_channel_completed(array $config, array $channel): void
{
    $token = $config['telegram']['bot_token'];
    $title = clean_string($channel['title'] ?? 'کانال');
    $target = (int) ($channel['target_joins'] ?? 0);
    $current = (int) ($channel['current_joins'] ?? 0);
    $totalJoined = (int) ($channel['total_joins'] ?? $current);
    $totalLeft = (int) ($channel['total_left'] ?? 0);

    $message = "کمپین جذب به پایان رسید 🎉\n"
        . sprintf("عنوان کانال: %s\n", $title)
        . sprintf("هدف کمپین: %d نفر\n", $target)
        . sprintf("اعضای باقی‌مانده از این کمپین: %d نفر\n", $current)
        . sprintf("کل اعضای جذب‌شده: %d نفر\n", $totalJoined)
        . sprintf("ترک کرده‌اند: %d نفر\n", $totalLeft)
        . "کانال از لیست اسپانسر حذف شد. در صورت نیاز کمپین جدید بساز.";

    $recipients = array_unique(array_merge(
        $config['telegram']['sudo_ids'] ?? [],
        [(int) ($channel['added_by'] ?? 0)]
    ));

    $adminGroupId = (int) ($config['telegram']['admin_group_id'] ?? 0);
    if ($adminGroupId !== 0) {
        $recipients[] = $adminGroupId;
    }

    foreach ($recipients as $adminId) {
        if ($adminId > 0) {
            telegram_send_message($token, $adminId, $message);
        }
    }
}

function update_force_channel_target(PDO $pdo, int $channelId, int $target, string $inviteLink, ?int $baselineCount = null): void
{
    if (!force_channel_supports_progress($pdo)) {
        update_force_channel_invite_link($pdo, $channelId, $inviteLink);
        return;
    }

    try {
        $query = 'UPDATE force_channels SET invite_link = :invite_link, target_joins = :target, current_joins = 0, total_joins = 0, total_left = 0, total_invites = 0';
        $params = [
            ':invite_link' => $inviteLink,
            ':target' => $target,
            ':id' => $channelId,
        ];

        if ($baselineCount !== null && $baselineCount > 0) {
            $query .= ', member_baseline = :baseline, member_snapshot = :snapshot';
            $params[':baseline'] = $baselineCount;
            $params[':snapshot'] = $baselineCount;
        } else {
            $query .= ', member_baseline = NULL, member_snapshot = NULL';
        }

        $query .= ' WHERE id = :id';

        $stmt = $pdo->prepare($query);
        $stmt->execute($params);
        $pdo->prepare('DELETE FROM force_channel_members WHERE channel_id = :id')->execute([':id' => $channelId]);
    } catch (Throwable $exception) {
        update_force_channel_invite_link($pdo, $channelId, $inviteLink);
    }
}

function reset_force_channel_progress(PDO $pdo, int $channelId): void
{
    if (!force_channel_supports_progress($pdo)) {
        update_force_channel_invite_link($pdo, $channelId, null);
        return;
    }

    try {
        $stmt = $pdo->prepare('UPDATE force_channels SET target_joins = NULL, current_joins = 0, total_joins = 0, total_left = 0, total_invites = 0, invite_link = NULL, member_baseline = NULL, member_snapshot = NULL WHERE id = :id');
        $stmt->execute([':id' => $channelId]);
        $pdo->prepare('DELETE FROM force_channel_members WHERE channel_id = :id')->execute([':id' => $channelId]);
    } catch (Throwable $exception) {
        update_force_channel_invite_link($pdo, $channelId, null);
    }
}

function delete_force_channel(PDO $pdo, array $config, int $channelId): ?array
{
    $channel = fetch_force_channel($pdo, $channelId);
    if (!$channel) {
        return null;
    }

    $chatId = (int) ($channel['chat_id'] ?? 0);

    $stmt = $pdo->prepare('DELETE FROM force_channels WHERE id = :id');
    $stmt->execute([':id' => $channelId]);

    if ($chatId !== 0 && !is_force_channel_bot($channel)) {
        try {
            telegram_request($config['telegram']['bot_token'], 'leaveChat', ['chat_id' => $chatId]);
        } catch (Throwable $exception) {
            bot_log($config['app']['log_file'], 'leave_chat_failed', [
                'chat_id' => $chatId,
                'error' => $exception->getMessage(),
            ]);
        }
    }

    return $channel;
}

function ensure_force_channel_progress_columns(PDO $pdo): void
{
    static $attempted = false;
    if ($attempted) {
        return;
    }
    $attempted = true;

    try {
        $pdo->query('SELECT 1 FROM force_channels LIMIT 1');
    } catch (Throwable $exception) {
        return;
    }

    $definitions = [
        'target_joins' => 'ALTER TABLE force_channels ADD COLUMN target_joins INT UNSIGNED DEFAULT NULL',
        'current_joins' => 'ALTER TABLE force_channels ADD COLUMN current_joins INT UNSIGNED NOT NULL DEFAULT 0',
        'member_baseline' => 'ALTER TABLE force_channels ADD COLUMN member_baseline INT UNSIGNED DEFAULT NULL',
        'member_snapshot' => 'ALTER TABLE force_channels ADD COLUMN member_snapshot INT UNSIGNED DEFAULT NULL',
        'total_joins' => 'ALTER TABLE force_channels ADD COLUMN total_joins INT UNSIGNED NOT NULL DEFAULT 0',
        'total_left' => 'ALTER TABLE force_channels ADD COLUMN total_left INT UNSIGNED NOT NULL DEFAULT 0',
        'total_invites' => 'ALTER TABLE force_channels ADD COLUMN total_invites INT UNSIGNED NOT NULL DEFAULT 0',
    ];

    foreach ($definitions as $column => $sql) {
        try {
            $stmt = $pdo->prepare('SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = "force_channels" AND COLUMN_NAME = :column');
            $stmt->execute([':column' => $column]);
            if ((int) $stmt->fetchColumn() === 0) {
                $pdo->exec($sql);
            }
        } catch (Throwable $exception) {
            // ignore, we will fall back gracefully later
        }
    }
}

function force_channel_supports_progress(PDO $pdo): bool
{
    static $supports = null;
    if ($supports !== null) {
        return $supports;
    }

    ensure_force_channel_progress_columns($pdo);

    try {
        $pdo->query('SELECT total_joins FROM force_channels LIMIT 0');
        $supports = true;
    } catch (Throwable $exception) {
        $supports = false;
    }

    return $supports;
}

function ensure_force_channel_member_structure(PDO $pdo): void
{
    static $checked = false;
    if ($checked) {
        return;
    }
    $checked = true;

    try {
        $pdo->query('SELECT 1 FROM force_channel_members LIMIT 0');
    } catch (Throwable $exception) {
        return;
    }

    $addedColumn = false;
    $columns = [
        'telegram_id' => "ALTER TABLE force_channel_members ADD COLUMN telegram_id BIGINT NOT NULL DEFAULT 0 AFTER user_id",
        'joined_at' => "ALTER TABLE force_channel_members ADD COLUMN joined_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP",
    ];

    foreach ($columns as $column => $sql) {
        try {
            $stmt = $pdo->prepare('SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = "force_channel_members" AND COLUMN_NAME = :column');
            $stmt->execute([':column' => $column]);
            if ((int) $stmt->fetchColumn() === 0) {
                $pdo->exec($sql);
                if ($column === 'telegram_id') {
                    $addedColumn = true;
                }
            }
        } catch (Throwable $exception) {
            // ignore
        }
    }

    if ($addedColumn) {
        try {
            $pdo->exec('UPDATE force_channel_members m INNER JOIN bot_users u ON u.id = m.user_id SET m.telegram_id = u.telegram_id WHERE m.telegram_id = 0');
        } catch (Throwable $exception) {
            // ignore
        }
    }

    $needsPrimary = false;
    try {
        $stmt = $pdo->query("SELECT COUNT(*) FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = 'force_channel_members' AND CONSTRAINT_NAME = 'PRIMARY' AND COLUMN_NAME = 'telegram_id'");
        $needsPrimary = ((int) $stmt->fetchColumn()) === 0;
    } catch (Throwable $exception) {
        $needsPrimary = true;
    }

    if ($needsPrimary) {
        try {
            $pdo->exec('ALTER TABLE force_channel_members DROP PRIMARY KEY');
        } catch (Throwable $exception) {
            // ignore
        }

        try {
            $pdo->exec('ALTER TABLE force_channel_members ADD PRIMARY KEY (channel_id, telegram_id)');
        } catch (Throwable $exception) {
            // ignore
        }
    }

    try {
        $pdo->exec('CREATE INDEX idx_force_member_channel ON force_channel_members (channel_id)');
    } catch (Throwable $exception) {
        // ignore if exists
    }
}

function find_bot_user_by_telegram_id(PDO $pdo, int $telegramId): ?array
{
    if ($telegramId === 0) {
        return null;
    }

    try {
        $stmt = $pdo->prepare('SELECT * FROM bot_users WHERE telegram_id = :telegram_id LIMIT 1');
        $stmt->execute([':telegram_id' => $telegramId]);
        $row = $stmt->fetch();
        return $row ?: null;
    } catch (Throwable $exception) {
        return null;
    }
}

function fetch_force_channel_member_count(array $config, array $channel): ?int
{
    if (is_force_channel_bot($channel)) {
        return null;
    }
    $token = $config['telegram']['bot_token'];
    $chatId = (int) ($channel['chat_id'] ?? 0);

    if ($chatId === 0) {
        return null;
    }

    try {
        $result = telegram_request($token, 'getChatMemberCount', [
            'chat_id' => $chatId,
        ]);
    } catch (Throwable $exception) {
        bot_log($config['app']['log_file'], 'fetch_member_count_failed', [
            'chat_id' => $chatId,
            'error' => $exception->getMessage(),
        ]);
        return null;
    }

    if (is_array($result)) {
        $count = (int) ($result['count'] ?? 0);
    } else {
        $count = (int) $result;
    }

    return $count > 0 ? $count : null;
}

function synchronize_force_channel_counters(PDO $pdo, array $config, array $channel): array
{
    if (is_force_channel_bot($channel)) {
        return $channel;
    }
    ensure_force_channel_member_structure($pdo);
    if (!force_channel_supports_progress($pdo)) {
        return $channel;
    }

    $channelId = (int) ($channel['id'] ?? 0);
    if ($channelId === 0) {
        return $channel;
    }

    $memberCount = fetch_force_channel_member_count($config, $channel);
    if ($memberCount === null) {
        return $channel;
    }

    $baseline = (int) ($channel['member_baseline'] ?? 0);
    $snapshot = (int) ($channel['member_snapshot'] ?? 0);
    $totalJoined = (int) ($channel['total_joins'] ?? 0);
    $totalLeft = (int) ($channel['total_left'] ?? 0);
    $current = (int) ($channel['current_joins'] ?? 0);

    if ($baseline === 0) {
        $baseline = $memberCount;
    }

    if ($snapshot === 0) {
        $snapshot = $baseline;
    }

    $difference = $memberCount - $snapshot;

    if ($difference > 0) {
        $totalJoined += $difference;
        $current += $difference;
    } elseif ($difference < 0) {
        $left = abs($difference);
        $totalLeft += $left;
        $current = max(0, $current - $left);
    }

    if ($totalJoined < $totalLeft) {
        $totalLeft = $totalJoined;
    }

    $current = max(0, $totalJoined - $totalLeft);

    try {
        $stmt = $pdo->prepare('UPDATE force_channels SET member_baseline = :baseline, member_snapshot = :snapshot, total_joins = :total_joins, total_left = :total_left, current_joins = :current WHERE id = :id');
        $stmt->execute([
            ':baseline' => $baseline,
            ':snapshot' => $memberCount,
            ':total_joins' => $totalJoined,
            ':total_left' => $totalLeft,
            ':current' => $current,
            ':id' => $channelId,
        ]);
    } catch (Throwable $exception) {
        bot_log($config['app']['log_file'], 'sync_force_channel_failed', [
            'channel_id' => $channelId,
            'error' => $exception->getMessage(),
        ]);
    }

    return fetch_force_channel($pdo, $channelId) ?? $channel;
}

function ensure_force_channel_invite_link(PDO $pdo, array $config, array $channel): array
{
    if (is_force_channel_bot($channel)) {
        return $channel;
    }
    $channelId = (int) ($channel['id'] ?? 0);
    if ($channelId === 0) {
        return $channel;
    }

    $target = (int) ($channel['target_joins'] ?? 0);
    $hasInvite = !empty($channel['invite_link']);

    if ($target <= 0 || $hasInvite) {
        return $channel;
    }

    $invite = create_force_channel_invite_link($pdo, $config, $channel, $target);
    if (!$invite) {
        return $channel;
    }

    return fetch_force_channel($pdo, $channelId) ?? $channel;
}

function notify_force_channel_invite_created(array $config, array $channel, string $inviteLink, ?int $memberLimit): void
{
    $token = $config['telegram']['bot_token'];
    $title = clean_string($channel['title'] ?? 'کانال');
    $limitText = $memberLimit !== null ? sprintf('%d نفر', $memberLimit) : 'بدون محدودیت مشخص';

    $message = "لینک جدید برای کمپین اسپانسر ساخته شد ✅\n"
        . sprintf("عنوان کانال: %s\n", $title)
        . sprintf("لینک: %s\n", $inviteLink)
        . sprintf("محدودیت لینک: %s", $limitText);

    $recipients = array_unique(array_merge(
        $config['telegram']['sudo_ids'] ?? [],
        [(int) ($channel['added_by'] ?? 0)]
    ));

    foreach ($recipients as $adminId) {
        if ($adminId > 0) {
            telegram_send_message($token, $adminId, $message);
        }
    }
}

function finalize_force_channel_if_completed(PDO $pdo, array $config, array $channel): ?array
{
    if (!$channel || is_force_channel_bot($channel)) {
        return $channel;
    }

    $target = (int) ($channel['target_joins'] ?? 0);
    $current = (int) ($channel['current_joins'] ?? 0);

    if ($target > 0 && $current >= $target) {
        notify_force_channel_completed($config, $channel);
        delete_force_channel($pdo, $config, (int) ($channel['id'] ?? 0));
        return null;
    }

    return $channel;
}

function send_broadcast_filter_menu(array $config, int $chatId): void
{
    $keyboard = make_inline_keyboard([
        [
            [
                'text' => 'همه کاربران 👥',
                'callback_data' => 'broadcast_filter:all',
            ]
        ],
        [
            [
                'text' => 'کاربران فعال روز 👥',
                'callback_data' => 'broadcast_filter:day',
            ], [
                'text' => 'کاربران فعال هفته 👥',
                'callback_data' => 'broadcast_filter:week',
            ]
        ],
        [
            [
                'text' => 'کاربران فعال ماه 👥',
                'callback_data' => 'broadcast_filter:month',
            ]
        ],
        [
            [
                'text' => 'بازگشت ↩️',
                'callback_data' => 'admin_menu',
            ]
        ],
    ]);

    telegram_send_message(
        $config['telegram']['bot_token'],
        $chatId,
        "کاربرای هدف رو انتخاب کن تا ارسال همگانی شروع بشه ✨",
        ['reply_markup' => $keyboard]
    );
}

function send_broadcast_options_menu(array $config, int $chatId, string $filter, int $ttlSeconds, bool $pinMessage, ?int $messageId = null): void
{
    $labels = [
        'all' => 'همه کاربران',
        'day' => 'کاربران فعال روز',
        'week' => 'کاربران فعال هفته',
        'month' => 'کاربران فعال ماه',
    ];
    $label = $labels[$filter] ?? $labels['all'];

    $ttlOptions = [
        0 => 'همیشه بماند',
        15 => '15 ثانیه',
        30 => '30 ثانیه',
        60 => '1 دقیقه',
        120 => '2 دقیقه',
        180 => '3 دقیقه',
        300 => '5 دقیقه',
        600 => '10 دقیقه',
        1800 => '30 دقیقه',
        3600 => '1 ساعت',
    ];

    $keyboard = [];
    $alwaysLabel = $ttlOptions[0];
    if ($ttlSeconds === 0) {
        $alwaysLabel = sprintf('✅ %s', $alwaysLabel);
    }
    $keyboard[] = [[
        'text' => $alwaysLabel,
        'callback_data' => 'broadcast_ttl:0',
    ]];

    $rows = [
        [15, 30, 60],
        [120, 180, 300],
        [600, 1800, 3600],
    ];

    foreach ($rows as $rowValues) {
        $row = [];
        foreach ($rowValues as $value) {
            $labelText = $ttlOptions[$value] ?? format_duration_label($value);
            if ($value === $ttlSeconds) {
                $labelText = sprintf('✅ %s', $labelText);
            }
            $row[] = [
                'text' => $labelText,
                'callback_data' => sprintf('broadcast_ttl:%d', $value),
            ];
        }
        $keyboard[] = $row;
    }

    $pinLabel = $pinMessage ? '🧷 سنجاق پیام: ✅' : '🧷 سنجاق پیام: ❌';
    $keyboard[] = [[
        'text' => $pinLabel,
        'callback_data' => 'broadcast_pin:toggle',
    ]];

    $keyboard[] = [[
        'text' => 'ادامه ✅',
        'callback_data' => 'broadcast_options_continue',
    ]];

    $keyboard[] = [
        [
            'text' => 'انصراف ❌',
            'callback_data' => 'broadcast_abort',
        ],
        [
            'text' => 'بازگشت ↩️',
            'callback_data' => 'broadcast_options_back',
        ],
    ];

    $text = "گیرندگان انتخاب شده: {$label}\n\nزمان نگهداری پیام و وضعیت سنجاق شدن را انتخاب کن ⏱";
    $replyMarkup = make_inline_keyboard($keyboard);

    if ($messageId !== null) {
        try {
            telegram_request($config['telegram']['bot_token'], 'editMessageText', [
                'chat_id' => $chatId,
                'message_id' => $messageId,
                'text' => $text,
                'reply_markup' => $replyMarkup,
            ]);
            return;
        } catch (Throwable $exception) {
            bot_log($config['app']['log_file'], 'broadcast_options_edit_failed', [
                'chat_id' => $chatId,
                'error' => $exception->getMessage(),
            ]);
        }
    }

    telegram_send_message(
        $config['telegram']['bot_token'],
        $chatId,
        $text,
        ['reply_markup' => $replyMarkup]
    );
}

function send_broadcast_message_prompt(array $config, int $chatId, string $filter, int $ttlSeconds, bool $pinMessage): void
{
    $labels = [
        'all' => 'همه کاربران',
        'day' => 'کاربران فعال روز',
        'week' => 'کاربران فعال هفته',
        'month' => 'کاربران فعال ماه',
    ];

    $label = $labels[$filter] ?? $labels['all'];

    $text = "متن یا پستی که می‌خوای به صورت همگانی ارسال بشه را بفرست 📨\n\n" .
        "گیرندگان انتخاب شده: {$label}";
    $text .= "\n⏱ حذف خودکار: " . format_duration_label($ttlSeconds);
    $text .= "\n📌 سنجاق پیام: " . ($pinMessage ? 'بله' : 'خیر');

    $keyboard = make_inline_keyboard([
        [
            [
                'text' => 'انصراف ❌',
                'callback_data' => 'broadcast_abort',
            ]
        ],
        [
            [
                'text' => 'بازگشت ↩️',
                'callback_data' => 'broadcast_options_back',
            ]
        ],
    ]);

    telegram_send_message(
        $config['telegram']['bot_token'],
        $chatId,
        $text,
        ['reply_markup' => $keyboard]
    );
}

function handle_admin_broadcast_message(PDO $pdo, array $config, array $message, array $state): bool
{
    $token = $config['telegram']['bot_token'];
    $chatId = (int) ($message['chat']['id'] ?? 0);
    if ($chatId === 0) {
        return false;
    }

    $filter = $state['pending_group_id'] ?? 'all';
    if (!in_array($filter, ['all', 'day', 'week', 'month'], true)) {
        $filter = 'all';
    }

    $ttlSeconds = max(0, (int) ($state['pending_code'] ?? 0));
    $pinMessage = (int) ($state['pending_position'] ?? 0) === 1;

    $text = isset($message['text']) ? trim($message['text']) : '';
    if ($text === '/cancel') {
        telegram_send_message($token, $chatId, 'ارسال همگانی لغو شد ❌');
        reset_admin_state($pdo, $chatId);
        send_admin_menu($config, $chatId);
        return true;
    }

    $sourceChatId = (int) ($message['chat']['id'] ?? 0);
    $sourceMessageId = (int) ($message['message_id'] ?? 0);
    if ($sourceMessageId === 0) {
        telegram_send_message($token, $chatId, 'نتونستم پیام رو شناسایی کنم. دوباره امتحان کن.');
        return true;
    }

    try {
        $job = create_broadcast_job($pdo, $config, $sourceChatId, $sourceMessageId, $chatId, $filter, $ttlSeconds, $pinMessage);
        if ($job === null) {
            reset_admin_state($pdo, $chatId);
            telegram_send_message($token, $chatId, 'هیچ کاربری برای این فیلتر پیدا نشد 😕');
            send_broadcast_filter_menu($config, $chatId);
            return true;
        }

        reset_admin_state($pdo, $chatId);

        $launched = launch_broadcast_worker($config, (int) $job['id']);
        if ($launched) {
            telegram_send_message($token, $chatId, 'ارسال همگانی شروع شد ✅', [
                'reply_markup' => make_inline_keyboard([
                    [[
                        'text' => 'بازگشت ↩️',
                        'callback_data' => 'admin_menu',
                    ]],
                ]),
            ]);
        } else {
        telegram_send_message($token, $chatId, 'ارسال همگانی در صف قرار گرفت ⏳');
        }

        return true;
    } catch (Throwable $exception) {
        bot_log($config['app']['log_file'], 'broadcast_job_create_failed', [
            'error' => $exception->getMessage(),
        ]);
        telegram_send_message($token, $chatId, 'ساخت ارسال همگانی با خطا مواجه شد ❗️\n' . $exception->getMessage());
    }

    reset_admin_state($pdo, $chatId);
    send_admin_menu($config, $chatId);
    return true;
}

function create_broadcast_job(PDO $pdo, array $config, int $sourceChatId, int $sourceMessageId, int $createdBy, string $filter, int $autoDeleteSeconds, bool $pinMessage): ?array
{
    $pdo->prepare('INSERT INTO broadcast_jobs (status, filter, source_chat_id, source_message_id, created_by, auto_delete_seconds, pin_message) VALUES (\'pending\', :filter, :source_chat_id, :source_message_id, :created_by, :auto_delete_seconds, :pin_message)')->execute([
        ':filter' => $filter,
        ':source_chat_id' => $sourceChatId,
        ':source_message_id' => $sourceMessageId,
        ':created_by' => $createdBy,
        ':auto_delete_seconds' => max(0, $autoDeleteSeconds),
        ':pin_message' => $pinMessage ? 1 : 0,
    ]);

    $jobId = (int) $pdo->lastInsertId();
    $total = queue_broadcast_recipients($pdo, $jobId, $filter);

    if ($total === 0) {
        $pdo->prepare("UPDATE broadcast_jobs SET status = 'completed', total_count = 0, finished_at = NOW() WHERE id = :id")
            ->execute([':id' => $jobId]);
        return null;
    }

    $pdo->prepare('UPDATE broadcast_jobs SET total_count = :total WHERE id = :id')
        ->execute([':total' => $total, ':id' => $jobId]);

    $job = fetch_broadcast_job($pdo, $jobId);
    if (!$job) {
        return null;
    }

    $text = build_broadcast_progress_text($job);
    $keyboard = build_broadcast_progress_keyboard($jobId, false);
    $response = telegram_send_message(
        $config['telegram']['bot_token'],
        $createdBy,
        $text,
        ['reply_markup' => $keyboard]
    );

    $pdo->prepare('UPDATE broadcast_jobs SET progress_chat_id = :chat_id, progress_message_id = :message_id WHERE id = :id')
        ->execute([
            ':chat_id' => $response['chat']['id'] ?? $createdBy,
            ':message_id' => $response['message_id'] ?? null,
            ':id' => $jobId,
        ]);

    bot_log($config['app']['log_file'], 'broadcast_job_created', [
        'job_id' => $jobId,
        'filter' => $filter,
        'total' => $total,
        'ttl_seconds' => $autoDeleteSeconds,
        'pin_message' => $pinMessage,
    ]);

    return fetch_broadcast_job($pdo, $jobId);
}

function queue_broadcast_recipients(PDO $pdo, int $jobId, string $filter): int
{
    $conditions = [
        'is_blocked = 0',
        'has_blocked_bot = 0',
        'telegram_id IS NOT NULL',
    ];

    $interval = null;
    if ($filter === 'day') {
        $interval = '1 DAY';
    } elseif ($filter === 'week') {
        $interval = '7 DAY';
    } elseif ($filter === 'month') {
        $interval = '30 DAY';
    }

    if ($interval !== null) {
        $conditions[] = "COALESCE(last_seen_at, joined_at) >= DATE_SUB(NOW(), INTERVAL {$interval})";
    }

    $sql = sprintf(
        'INSERT INTO broadcast_queue (job_id, user_id) SELECT :job_id, id FROM bot_users WHERE %s',
        implode(' AND ', $conditions)
    );

    $stmt = $pdo->prepare($sql);
    $stmt->execute([':job_id' => $jobId]);

    $countStmt = $pdo->prepare('SELECT COUNT(*) FROM broadcast_queue WHERE job_id = :job_id');
    $countStmt->execute([':job_id' => $jobId]);
    return (int) $countStmt->fetchColumn();
}

function fetch_broadcast_job(PDO $pdo, int $jobId): ?array
{
    $stmt = $pdo->prepare('SELECT * FROM broadcast_jobs WHERE id = :id LIMIT 1');
    $stmt->execute([':id' => $jobId]);
    $row = $stmt->fetch();
    return $row ?: null;
}

function update_broadcast_progress_message(PDO $pdo, array $config, int $jobId, bool $finished): void
{
    $job = fetch_broadcast_job($pdo, $jobId);
    if (!$job) {
        return;
    }

    if (empty($job['progress_chat_id']) || empty($job['progress_message_id'])) {
        return;
    }

    $text = build_broadcast_progress_text($job);
    $keyboard = build_broadcast_progress_keyboard($jobId, $finished);

    try {
        telegram_request($config['telegram']['bot_token'], 'editMessageText', [
            'chat_id' => (int) $job['progress_chat_id'],
            'message_id' => (int) $job['progress_message_id'],
            'text' => $text,
            'reply_markup' => $keyboard,
        ]);
    } catch (Throwable $exception) {
        bot_log($config['app']['log_file'], 'broadcast_progress_edit_failed', [
            'job_id' => $jobId,
            'error' => $exception->getMessage(),
        ]);
    }
}

function launch_broadcast_worker(array $config, int $jobId): bool
{
    if (!function_exists('exec')) {
        bot_log($config['app']['log_file'], 'broadcast_worker_exec_disabled', [
            'job_id' => $jobId,
            'php_binary' => PHP_BINARY,
            'disable_functions' => ini_get('disable_functions'),
            'safe_mode' => ini_get('safe_mode'),
            'open_basedir' => ini_get('open_basedir'),
        ]);
        return false;
    }

    $php = $config['app']['php_binary'] ?? '/usr/bin/php';
    $worker = __DIR__ . '/broadcast_worker.php';

    $command = sprintf(
        'nohup %s %s %d > /dev/null 2>&1 &',
        escapeshellcmd($php),
        escapeshellarg($worker),
        $jobId
    );

    $result = @exec($command);

    bot_log($config['app']['log_file'], 'broadcast_worker_launch_attempt', [
        'job_id' => $jobId,
        'command' => $command,
        'result' => $result,
        'php_binary' => $php,
        'disable_functions' => ini_get('disable_functions'),
        'safe_mode' => ini_get('safe_mode'),
        'open_basedir' => ini_get('open_basedir'),
    ]);

    return true;
}

function cancel_broadcast_job(PDO $pdo, int $jobId): bool
{
    $stmt = $pdo->prepare("UPDATE broadcast_jobs SET status = 'cancelled', finished_at = NOW() WHERE id = :id AND status IN ('pending','running')");
    $stmt->execute([':id' => $jobId]);
    return $stmt->rowCount() > 0;
}


